package com.livescribe.ext.util;

import java.util.Enumeration;

/**
 * <code>PriorityQueue</code> implements the standard Java SE class
 * Underlying implementation is a standard heap
 *
 * @author slynggaard
 */


public class PriorityQueue implements Cloneable, Queue {

	private Object[] heap;
	private int size;

	public PriorityQueue() {
		this.clear();
	}

	public PriorityQueue(int initialCapacity ) {
		heap = new Object[initialCapacity];
	}

	public PriorityQueue(Collection c) {
		this.clear();
		Iterator itr = c.iterator();
		while( itr.hasNext() ) {
			Object o = itr.next();
			this.offer(o);
		}
	}

	public PriorityQueue(Enumeration en) {
		this.clear();
		while( en.hasMoreElements() ) {
			this.offer(en.nextElement());
		}
	}

	public PriorityQueue(Object[] o) {
		this.clear();
		for( int i = 0 ; i < o.length ; i++ ) {
			this.offer(o[i]);
		}
	}

	/**
	 *  Inserts object into queue
	 * @param element
	 * @return
	 */

	public boolean add(Object element) {
		 return offer(element);
	}

	/**
	 * Clears queue
	 */

	 public void clear() {
		 heap = new Object[4];
		 size = 0;
	 }

	 /**
	  * Returns true if object exists in queue
	  * @param o
	  * @return
	  */

	 public boolean contains(Object o) {
		for(int i = 0 ; i < size ; i++ ) {
			if ( o.equals(heap[i]) )
				return true;
		}
		return false;
	 }

	 private class QueueIterator implements Iterator {

		private PriorityQueue queue;
		private int location;

		public QueueIterator( PriorityQueue queue ) {
			location = 0;
			this.queue = queue;
		}

		public boolean hasNext() {
			if( location >= queue.size ) {
				return false;
			}

			return true;
		}

		public Object next() {
			if( location >= size ) {
				return null;
			}
			return queue.heap[location++];
		}

		public void remove() {
			queue.removeAtIndex(location);
		}

	 }

	 public Iterator iterator() {
		 QueueIterator itr = new QueueIterator( this );
		 return itr;

	 }

	 private void checkSize() {
		 int newsize;

		 // increase size?
		 if( size == heap.length ) {
			if( heap.length < 256 ) {
				newsize = heap.length*2;
			} else {
				newsize = heap.length+256;
			}
		 // shrink it?
		 } else if( size + 512 < heap.length ){
			 newsize = heap.length-256;
		 }
		 else {
			 return;
		 }

		 Object[] newheap = new Object[newsize];
		 java.lang.System.arraycopy(heap, 0,newheap, 0, size);
		 heap = newheap;
	 }

	 private void swap(int i1, int i2) {
		 Object temp =  heap[i1];
		 heap[i1] = heap[i2];
		 heap[i2] = temp;
	 }

	 /**
	  * Inserts object into queue
	  */

	 public boolean offer(Object element) {
		 int pos = size;

		 if( !(element instanceof Comparable) ) {
			 return false;
		 }
		 checkSize();

		 heap[pos] = element;

		 while( pos > 0 && ((Comparable)heap[pos]).compareTo( heap[(pos-1)>>1] ) > 0 ) {
			int pos2= (pos-1)>>1;
			swap( pos, pos2 );
			pos = pos2;
		 }

		 size++;

		 return true;
	 }

	 /**
	  * Looks at first element but does not remove
	  * @return
	  */

	 public Object peek() {

		 if( size == 0 ) {
			 return null;
		 }

		 return heap[0];
	 }

	 /**
	  * Main removal function
	  * @param i
	  * @return
	  */

	 private Object removeAtIndex( int i ) {
		 Object ret = heap[i];

		 if( size == 0 ) {
			 return null;
		 }

		 size--;
		 heap[i] = heap[size];

		 int pos = i;


		 while( pos >= 0 ) {
			 int newpos = pos*2;

			 if( newpos+2 > size  || ((Comparable)heap[newpos+1]).compareTo(heap[newpos+2]) > 0 ) {
				 if( newpos+1 > size )
					 break;					// get us out of here

				 newpos +=1;
			 }
			 else {
				 newpos +=2;
			 }

			 if( ((Comparable)heap[pos]).compareTo(heap[newpos])  < 0 ) {
				 swap(pos,newpos);
				 pos = newpos;
			 }
			 else {
				 pos = -1;
			 }
		 }

		 checkSize();
		 return ret;
	 }

	 /**
	  * Gets the first element from queue
	  * @return
	  */
		public Object poll() {
			return removeAtIndex(0);
		}

		/**
		 * Retrieves but does not remove the first element in the queue
		 */

		public Object element() {
			return peek();
		}

		/**
		 * Removes the first element from the queue
		 */

		public Object remove() {
			return poll();
		}


	 /**
	  * Removes a single object from the queue
	  * @param o
	  * @return
	  */
	 public boolean remove(Object o) {

		 int i;
		 for( i = 0 ; i < size ; i++) {
			 if( heap[i].equals(o) ) {
				 break;
			 }
		 }

		 if( i == size ) {
			 return false;
		 }

		 removeAtIndex(i);

		 return true;
	 }

	 /**
	  * Returns the number of objects
	  * @return
	  */
	 public int size() {
		 return size;
	 }

	 /**
	  * Returns an array of all the objects in the queue
	  * @return
	  */
	 public Object[] toArray() {
		 Object[] o = new Object[heap.length];
		 System.arraycopy(heap, 0, o, 0, heap.length);
		 return o;
	 }

	 /**
	  * Create a shallow copy of the array
	  * @return
	  */

	 public Object clone() {
		 PriorityQueue res = new PriorityQueue();
		 res.heap  = new Object[heap.length];
		 System.arraycopy(heap, 0, res.heap , 0, heap.length);
		 res.size = size;
		 return res;
	 }


	/**
	 * Adds all elements of the collection to the priority queue
	 */

	public boolean addAll(Collection c) {
		if( c.isEmpty()) {
			return false;
		}

		Iterator itr = c.iterator();
		while( itr.hasNext() ) {
			this.offer(itr.next());
		}

		return true;
	}

	/**
	 * Check if priority queue contains all the objects in the collection
	 */

	public boolean containsAll(Collection c) {

		Iterator itr = c.iterator();
		while( itr.hasNext() ) {
			if( !this.contains(itr.next()) ) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Check if the queue is empty
	 */
	public boolean isEmpty() {
		return size == 0;
	}

	/**
	 * Remove all objects in the collection from the priority queue
	 */
	public boolean removeAll(Collection c) {
		Iterator itr = c.iterator();
		boolean changed = false;
		while( itr.hasNext() ) {
			if( this.remove(itr.next()) ) {
				changed = true;
			}
		}
		return changed;
	}

	/**
	 * Remove all but the objects in the PriorityQueue but the objects in the collection
	 */
	public boolean retainAll(Collection c) {
		boolean changed = false;
		Iterator itr = this.iterator();
		while( itr.hasNext() ) {
			Object o = itr.next();
			if( !c.contains(o) ) {
				itr.remove();
				changed = true;
			}
		}
		return changed;
	}
}
