/*
 * Date: 1-Nov-2009
 * Time: 8:57:47 PM
 * (c) 2010 Livescribe, Inc.
 */
package com.livescribe.ext.util;

import com.livescribe.penlet.Logger;

/**
 * A convenience class that provides access to the penlet's {@link Logger}
 * instance.  It is designed as a set of static methods so that a
 * {@link Logger} instance does not need to be passed around between classes
 * that wish to use the log.
 * <p>
 * In addition to the set of debug/info/warn/error/test outputs, the "out"
 * methods provide access to <code>System.out</code>.  Also, a set of
 * "tracing" methods provide nicely formatted debug output containing the
 * class name and method name.  Note, however, that the class name may be
 * obfuscated in certain projects.</p>
 * <p>
 * Note that the {@link Logger} instance must be set via
 * {@link #setLogger(Logger)} before any of its methods can be used.</p>
 *
 * @see Logger
 * @author Shawn Silverman
 */
public final class Log {
    private static Logger logger;

    /**
     * Disallow instantiation.
     */
    private Log() { }

    /**
     * Sets the current logger.  The logger can be set to <code>null</code>.
     *
     * @param logger the new {@link Logger} instance, or <code>null</code> to
     *               unset it.
     */
    public static synchronized void setLogger(Logger logger) {
        Log.logger = logger;
    }

    /**
     * Writes an <em>debug</em> message to the log.
     *
     * @param msg write this message
     */
    public static synchronized void debug(String msg) {
        if (logger != null) {
            logger.debug(msg);
        }
    }

    /**
     * Writes an <em>info</em> message to the log.
     *
     * @param msg write this message
     */
    public static synchronized void info(String msg) {
        if (logger != null) {
            logger.info(msg);
        }
    }

    /**
     * Writes an <em>warn</em> message to the log.
     *
     * @param msg write this message
     */
    public static synchronized void warn(String msg) {
        if (logger != null) {
            logger.warn(msg);
        }
    }

    /**
     * Writes an <em>error</em> message to the log.
     *
     * @param msg write this message
     */
    public static synchronized void error(String msg) {
        if (logger != null) {
            logger.error(msg);
        }
    }

    /**
     * Writes an <em>test</em> message to the log.
     *
     * @param msg write this message
     */
    public static synchronized void test(String msg) {
        if (logger != null) {
            logger.test(msg);
        }
    }

    /**
     * Sends the output to {@link System#out}.
     *
     * @param msg the message
     */
    public static void out(String msg) {
        System.out.println(msg);
    }

    /**
     * Sends the output to {@link System#out}.
     *
     * @param clss the class
     * @param method the method name
     * @param msg the message
     */
    public static void out(Class clss, String method, String msg) {
        System.out.println(buildMsg(clss, method, msg));
    }

    /**
     * Sends the output to {@link System#out}.
     *
     * @param className the class name
     * @param method the method name
     * @param msg the message
     */
    public static void out(String className, String method, String msg) {
        System.out.println(buildMsg(className, method, msg));
    }

    /**
     * Builds a message string from the class, method, and message.  This may
     * return <code>null</code> if all three parameters are <code>null</code>.
     * <p>
     * This calls {@link #buildMsg(String, String, String)}.</p>
     *
     * @param clss the class
     * @param method the method name
     * @param msg the message
     * @return a string built from the given parameters.
     */
    private static String buildMsg(Class clss, String method, String msg) {
        return buildMsg(clss != null ? clss.getName() : null, method, msg);
    }

    /**
     * Builds a message string from the class name, method, and message.  This
     * may return <code>null</code> if all three parameters are
     * <code>null</code>.  All of the class-method-message debugging methods
     * use this.
     * <p>
     * The message will be prefixed with
     * "<em>SimpleClassName</em>.<em>method</em>:&nbsp;".</p>
     *
     * @param className the class name
     * @param method the method name
     * @param msg the message
     * @return a string built from the given parameters.
     */
    private static String buildMsg(String className, String method, String msg) {
        // An edge case that avoids string buffer creation

        if (className == null && method == null) {
            return msg;
        }

        // Create a string buffer on the assumption that the message is
        // not null most of the time

        StringBuffer buf = new StringBuffer();

        if (className == null) {
            // NOTE method != null

            buf.append(method);
        } else {
            String name = className;

            int dotIndex = name.lastIndexOf('.');
            if (dotIndex >= 0) {
                name = name.substring(dotIndex + 1);
            }

            buf.append(name);

            if (method != null) {
                buf.append('.').append(method);
            }
        }

        // Possibly append the message

        if (msg != null) {
            buf.append(": ").append(msg);
        }

        return buf.toString();
    }

    /**
     * Writes a debug message to the log and prefixes the message with
     * "<em>SimpleClassName</em>.<em>method</em>:&nbsp;".
     *
     * @param clss the class
     * @param method the method name
     * @param msg the debug message
     */
    public static void debug(Class clss, String method, String msg) {
        debug(buildMsg(clss, method, msg));
    }

    /**
     * Writes a debug message to the log and prefixes the message with
     * "<em>SimpleClassName</em>.<em>method</em>:&nbsp;".
     *
     * @param className the class name
     * @param method the method name
     * @param msg the debug message
     */
    public static void debug(String className, String method, String msg) {
        debug(buildMsg(className, method, msg));
    }

    /**
     * Writes an info message to the log and prefixes the message with
     * "<em>SimpleClassName</em>.<em>method</em>:&nbsp;".
     *
     * @param clss the class
     * @param method the method name
     * @param msg the info message
     */
    public static void info(Class clss, String method, String msg) {
        info(buildMsg(clss, method, msg));
    }

    /**
     * Writes an info message to the log and prefixes the message with
     * "<em>SimpleClassName</em>.<em>method</em>:&nbsp;".
     *
     * @param className the class name
     * @param method the method name
     * @param msg the info message
     */
    public static void info(String className, String method, String msg) {
        info(buildMsg(className, method, msg));
    }

    /**
     * Writes a warning to the log and prefixes the message with
     * "<em>SimpleClassName</em>.<em>method</em>:&nbsp;".
     *
     * @param clss the class
     * @param method the method name
     * @param msg the warning message
     */
    public static void warn(Class clss, String method, String msg) {
        warn(buildMsg(clss, method, msg));
    }

    /**
     * Writes a warning to the log and prefixes the message with
     * "<em>SimpleClassName</em>.<em>method</em>:&nbsp;".
     *
     * @param className the class name
     * @param method the method name
     * @param msg the warning message
     */
    public static void warn(String className, String method, String msg) {
        warn(buildMsg(className, method, msg));
    }

    /**
     * Writes an error to the log and prefixes the message with
     * "<em>SimpleClassName</em>.<em>method</em>:&nbsp;".
     *
     * @param clss the class
     * @param method the method name
     * @param msg the error message
     */
    public static void error(Class clss, String method, String msg) {
        error(buildMsg(clss, method, msg));
    }

    /**
     * Writes an error to the log and prefixes the message with
     * "<em>SimpleClassName</em>.<em>method</em>:&nbsp;".
     *
     * @param className the class name
     * @param method the method name
     * @param msg the error message
     */
    public static void error(String className, String method, String msg) {
        error(buildMsg(className, method, msg));
    }

    /**
     * Writes a test message to the log and prefixes the message with
     * "<em>SimpleClassName</em>.<em>method</em>:&nbsp;".
     *
     * @param clss the class
     * @param method the method name
     * @param msg the test message
     */
    public static void test(Class clss, String method, String msg) {
        test(buildMsg(clss, method, msg));
    }

    /**
     * Writes a test message to the log and prefixes the message with
     * "<em>SimpleClassName</em>.<em>method</em>:&nbsp;".
     *
     * @param className the class name
     * @param method the method name
     * @param msg the test message
     */
    public static void test(String className, String method, String msg) {
        test(buildMsg(className, method, msg));
    }
}
