/*
 * Date: Apr 15, 2010
 * Time: 12:36:55 PM
 * (c) 2010 Livescribe, Inc.
 */
package com.livescribe.ext.util;

/**
 * Implementation of an exclusive lock.  This is used by more than one thread
 * to share a resource.  The semantics are the same as Java mutexes.  In other
 * words, the lock is reentrant.  A thread can reacquire the lock any number
 * of times without blocking.
 * <p>
 * When acquiring a lock, it is a good idea to ensure it is released by
 * putting it in a <em>finally</em> clause.</p>
 * <p>
 * Note: This uses the current instance for the wait/notify object.</p>
 *
 * @since 2.4
 *
 * @author Shawn Silverman
 */
public class Lock {
    private Thread owner;

    /**
     * Creates a new lock object.
     */
    public Lock() {
        super();
    }

    /**
     * Tries to acquire the lock.  The method blocks until the lock is
     * acquired.  This returns right away if the thread already owns the lock.
     *
     * @throws InterruptedException if this thread has been interrupted.
     */
    public synchronized void acquireLock() throws InterruptedException {
        if (owner == Thread.currentThread()) {
            return;
        }

        while (owner != null) {
            wait();
        }
        owner = Thread.currentThread();
    }

    /**
     * Tries to acquire the lock.  The method blocks until the lock is
     * acquired, or until the timeout expires.  This returns right away if the
     * thread already owns the lock.
     * <p>
     * As in the behaviour for {@link Object#wait(long)}, a timeout of zero
     * indicates that we should wait forever until notified or interrupted.</p>
     *
     * @param timeout the maximum time to wait, in ms
     * @return whether the lock was acquired before the timeout elapsed.
     * @throws InterruptedException if this thread was interrupted while
     *         waiting to acquire the lock.
     * @throws IllegalArgumentException if the timeout value is negative.
     */
    public synchronized boolean acquireLock(long timeout)
        throws InterruptedException
    {
        // Check the argument

        if (timeout < 0) {
            throw new IllegalArgumentException("Negative timeout: " + timeout);
        }

        // A zero timeout means wait forever

        if (timeout == 0) {
            acquireLock();

            return true;
        } else {
            if (owner == Thread.currentThread()) {
                return true;
            }

            long end = System.currentTimeMillis() + timeout;
            while (owner != null) {
                timeout = end - System.currentTimeMillis();
                if (timeout <= 0) {
                    break;
                }

                wait(timeout);
            }

            // Test the state

            if (owner != null) {
                return false;
            } else {
                owner = Thread.currentThread();
                return true;
            }
        }
    }

    /**
     * Releases the lock if the current thread is the owner.  All threads are
     * woken up to compete for the lock.
     */
    public synchronized void releaseLock() {
        // Don't allow other threads to release a lock they don't own

        if (owner == Thread.currentThread()) {
            owner = null;

            notifyAll();
        }
    }
}
