package com.livescribe.ext.util;


import java.util.Enumeration;
import java.util.NoSuchElementException;


/**
 * <code>LinkedList</code> implements the standard Java SE class
 *
 * @author slynggaard
 */

public class LinkedList implements Cloneable, List {

	private int size;
	private Node head;

	private class Node {
		protected Node succ;
		protected Node pred;
		protected Object data;

		public Node( Object data) {
			this.data = data;
			this.succ = this;
			this.pred = this;
		}

		public Node( Node succ, Node pred, Object data) {
			this.data = data;
			this.succ = succ;
			this.pred = pred;
		}

		public void InsertBefore( Node node ) {
			this.pred.succ = node;
			node.pred = this.pred;

			node.succ = this;
			this.pred = node;
		}

		public Node Remove() {
			Node n = this;
			this.succ.pred = this.pred;
			this.pred.succ = this.succ;
			return n;
		}
	}


	public LinkedList() {
		this.clear();
	}

	public LinkedList(Collection c) {
		this.clear();
		Iterator itr = c.iterator();
		while( itr.hasNext() ) {
			Object o = itr.next();
			this.addLast(o);
		}
	}

	public LinkedList(Enumeration en) {
		this.clear();
		while( en.hasMoreElements() ) {
			this.addLast(en.nextElement());
		}
	}

	public LinkedList(Object[] o) {
		this.clear();
		for( int i = 0 ; i < o.length ; i++ ) {
			this.offer(o[i]);
		}
	}

	/**
	 * Inserts supplied object at the specified index
	 * @param index
	 * @param element
	 */

	public void add(int index, Object element) {

		if( index < 0 || index > size ) {
			throw new IndexOutOfBoundsException();
		}

		Node current = head.succ;
		while( index-- > 0 ) {
			current = current.succ;
		}

		size++;

		current.InsertBefore(new Node(element));
	}

	/**
	 * Adds object to the end of the list
	 * @param o
	 * @return
	 */

	public boolean add(Object o) {
		size++;
		head.InsertBefore(new Node(o));
		return true;
	}

	/**
	 * Appends collection to the list
	 * @param c
	 * @return
	 */

	public boolean addAll(Collection c) {
		int size = c.size();
		boolean changed = false;
		Iterator itr = c.iterator();
		for(int i = 0 ; i < size ; i++ ) {
			Object o = itr.next();
			this.addLast(o);
			changed = true;
		}

		return changed;
	}

	/**
	 * Adds collection at index
	 * @param index
	 * @param c
	 * @return
	 */

	public boolean addAll(int index, Collection c) {
		if( index < 0 || index > size ) {
			throw new IndexOutOfBoundsException();
		}

		Object[] objs = c.toArray();

		if( objs.length > 0 ) {
			for(int i = 0 ; i < objs.length ; i++ ) {
				this.add(index++, objs[i]);
			}
			return true;
		}

		return false;
	}

	/**
	 * Returns true if the list contains all objects in the collection
	 */

	public boolean containsAll(Collection c) {
		Iterator itr = c.iterator();
		while( itr.hasNext() ) {
			Object o = itr.next();
			if( !this.contains(o) ) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Returns true if this collection is empty
	 */

	public boolean isEmpty() {
		if( size == 0 ) {
			return true;
		}
		return false;
	}


	/**
	 * Remove all objects in the list that is part of the collection
	 */

	public boolean removeAll(Collection c) {
		boolean changed = false;
		Iterator itr = c.iterator();
		while( itr.hasNext() ) {
			Object o = itr.next();
			if( this.remove(o)) {
				changed = true;
			}
		}

		return changed;
	}

	/**
	 * Removes all the objects in the list BUT the objects in the collection
	 */

	public boolean retainAll(Collection c) {
		boolean changed = false;
		Iterator itr = this.iterator();
		while( itr.hasNext() ) {
			Object o = itr.next();
			if( !c.contains(o) ) {
				itr.remove();
				changed = true;

			}
		}
		return changed;
	}

	/**
	 * Inserts object in the beginning of the list
	 * @param o
	 */

	public void addFirst(Object o) {
		size++;
		head.succ.InsertBefore(new Node(o));
	}

	/**
	 * Appends object to the end of list
	 * @param o
	 */
	public void addLast(Object o) {
		size++;
		head.InsertBefore(new Node(o));
	}

	/**
	 * Appends object to the end of list
	 * @param o
	 */

	public void offer(Object o) {
		addLast(o);
	}

	/**
	 * Removes all objects from the list
	 */
	public void clear() {
		head = new Node(null);
		size = 0;
	}

	/**
	 * Produces a shallow copy of the list
	 */
	public Object clone() {
		LinkedList ret = new LinkedList();
		Iterator itr = this.iterator();
		while( itr.hasNext() ) {
			ret.addLast(itr.next());
		}
		return ret;
	}

	/**
	 * Returns if the object exist in the list or not
	 * @param o
	 * @return
	 */
	public boolean contains(Object o) {
		Node current = head.succ;
		for( int i = 0 ; i < size ; i++) {
			if( (o == null && current.data ==  null ) || current.data.equals(o) ) {
				return true;
			}
			current = current.succ;
		}

		return false;
	}

	/**
	 * Gets the object at index
	 * @param index
	 * @return
	 */
	public Object get(int index) {

		if( index < 0 || index > size ) {
			throw new IndexOutOfBoundsException();
		}
		Node current = head.succ;
		while( index-- > 0 ) {
			current = current.succ;
		}

		return current.data;
	}

	/**
	 * Gets the first element
	 * @return
	 */
	public Object getFirst() {
		if( size == 0) {
			throw new NoSuchElementException();
		}
		return head.succ.data;
	}

	/**
	 * Peeks at the first element
	 * @return
	 */
	public Object peek() {
		return head.succ.data;
	}

	/**
	 * Peeks at the first element
	 * @return
	 */
	public Object element() {
		return head.succ.data;
	}
	/**
	 * Gets the last (tail) element
	 * @return
	 */
	public Object getLast() {
		if( size == 0) {
			throw new NoSuchElementException();
		}
		return head.pred.data;
	}

	/**
	 * Gets the index of the object. If object does no exist function returns -1
	 * @param o
	 * @return
	 */
	public int indexOf(Object o) {
		Node current = head.succ;
		for( int i = 0 ; i < size ; i++) {
			if( (o == null && current.data == null ) || current.data.equals(o) ) {
				return i;
			}
			current = current.succ;
		}

		return -1;
	}
	/**
	 * Returns the last occurrence of object in list. If object does not exist the function returns -1
	 */
	public int lastIndexOf(Object o) {
		Node current = head.pred;
		for( int i = 0 ; i < size ; i++) {
			if( (o == null && current.data == null ) || current.data.equals(o) ) {
				return size-i-1;
			}
			current = current.pred;
		}

		return -1;
	}

	/**
	 * Returns a ListIterator object to the list starting with object at index
	 * @param index
	 * @return
	 */


	public ListIterator listIterator(int index) {
		return new LinkedListIterator( index);
	}

	/**
	 * Removes the object at index
	 * @param index
	 * @return
	 */

	public Object remove(int index) {
		if( index < 0 || index > size ) {
			throw new IndexOutOfBoundsException();
		}

		Node current = head.succ;
		while( index-- > 0 ) {
			current = current.succ;
		}

		size--;

		return current.Remove().data;
	}

	/**
	 * Removes the first occurence of the object from the list
	 * @param o
	 * @return
	 */
	public boolean remove(Object o) {
		Node current = head.succ;
		for( int i = 0 ; i < size ; i++) {
			if( current.data == o ) {
				current.Remove();
				size--;
				return true;
			}
			current = current.succ;
		}

		return false;
	}

	/**
	 * Removes the first object from the list
	 * @return
	 */

	public Object removeFirst() {
		if( size == 0) {
			throw new NoSuchElementException();
		}

		Object ret = head.succ.Remove().data;
		size--;

		return 	ret;
	}

	/**
	 * Removes the first object from the list
	 * @return
	 */
	public Object remove() {
		return removeFirst();
	}

	/**
	 * Removes the first object from the list
	 * @return
	 */
	public Object poll() {
		if( size == 0) {
			return null;		// Yeah this is basically the same function as removeFirst but exception handling is different...
		}

		Object ret = head.succ.Remove().data;
		size--;

		return 	ret;
	}


	/**
	 * Removes the last (tail) object from the list
	 * @return
	 */
	public Object removeLast() {
		if( size == 0) {
			throw new NoSuchElementException();
		}

		size--;

		return 	head.pred.Remove().data;
	}

	/**
	 * Replace the object at index with the supplied object
	 * @param index
	 * @param element
	 * @return
	 */
	public Object set(int index, Object element) {
		if( index < 0 || index > size ) {
			throw new IndexOutOfBoundsException();
		}

		Node current = head.succ;
		while( index-- > 0 ) {
			current = current.succ;
		}

		Object ret = current.data;
		current.data = element;
		return ret;
	}

	/**
	 * Returns the number of objects in the list
	 * @return
	 */
	public int size() {
		return size;

	}

	/**
	 * Returns an array containing the objects in the list
	 * @return
	 */
	public Object[] toArray() {
		Object[] ret = new Object[size];

		Node current = head.succ;
		for( int i = 0 ; i < size ; i++) {
			ret[i] = current.data;
			current = current.succ;
		}

		return ret;
	}

	/**
	 * Equals implementation
	 */

	public boolean equals( Object o ) {
		if( !(o instanceof Collection) ){
			return false;
		}
		Collection c = (Collection)o;
		Iterator itr1 = this.iterator();
		Iterator itr2 = c.iterator();
		while( itr1.hasNext() && itr2.hasNext() ) {
			Object o1 = itr1.next();
			Object o2 = itr2.next();
			if( o1 != null || o2 != null ) {
				if( !o1.equals(o2) ) {
					return false;
				}
			}
		}

		if( itr1.hasNext() || itr2.hasNext() ) {
			return false;
		}

		return true;
	}

	/**
	 * Returns a string of this list
	 */

	public String toString() {
		StringBuffer strb = new StringBuffer();
		strb.append("{");
		Iterator itr = this.iterator();
		while( itr.hasNext() ) {
			strb.append( itr.next().toString() ) ;
			if( itr.hasNext() ) {
				strb.append( ", ");
			}
		}
		strb.append("}");
		return strb.toString();
	}
	/**
	 * Gets an iterator to the list
	 */
	public Iterator iterator() {
		return new LinkedListIterator( 0 );
	}

	/**
	 * Gets an iterator to the list
	 */

	public ListIterator listIterator() {
		return new LinkedListIterator( 0 );
	}

	/**
	 * Returns a new Sublist
	 */

	public List subList(int fromIndex, int toIndex) {
		LinkedList l = new LinkedList();
		ListIterator itr = new LinkedListIterator(fromIndex );

		toIndex-=fromIndex;
		while( itr.hasNext() && toIndex-- > 0 ) {
			l.addLast( itr.next() );
		}

		return l;
	}


	/**
	 * Sorts the elements. They must implement Comparable interface
	 */

	public void sort() {

		if( this.size < 2 ) {
			return;
		}

		LinkedList l1 = (LinkedList)this.subList(0, this.size>>1);
		LinkedList l2 = (LinkedList)this.subList((this.size>>1), this.size);
		l1.sort();
		l2.sort();

		this.clear();

		ListIterator itr1 = l1.new LinkedListIterator();
		ListIterator itr2 = l2.new LinkedListIterator();

		Object o1 = itr1.next();
		Object o2 = itr2.next();

		while( itr1.hasNext() && itr2.hasNext() ) {
			if( ((Comparable)o1).compareTo(o2) > 0 ) {
				this.addLast(o1);
				o1 = itr1.next();
			} else {
				this.addLast(o2);
				o2 = itr2.next();
			}

		}

		while( itr1.hasNext() ) {
			if( o2 == null || ((Comparable)o1).compareTo(o2) > 0 ) {
				this.addLast( o1 );
				o1 = itr1.next();
			} else {
				this.add( o2 );
				o2 = null;
			}
		}

		while( itr2.hasNext() ) {
			if( o1 == null || ((Comparable)o2).compareTo(o1) > 0 ) {
				this.addLast( o2 );
				o2 = itr2.next();
			} else {
				this.add( o1 );
				o1 = null;
			}
		}

		if( o1 != null && o2 != null ) {
			if( ((Comparable)o1).compareTo(o2) > 0 )  {
				this.addLast( o1 );
				o1 = null;
			} else {
				this.addLast( o2 );
				o2 = null;
			}
		}

		if( o2 != null ) {
			this.add( o2 );
		}

		if( o1 != null ) {
			this.add( o1 );
		}

	}

	private class LinkedListIterator implements Iterator, ListIterator{

		private Node curr;
		private int location;

		public LinkedListIterator() {
			location = 0;
			curr = head;
		}

		public LinkedListIterator( int index ) {
			if( index < 0 || index > size ) {
				throw new IndexOutOfBoundsException();
			}

			location = index;

			curr = head;
			while( index-- > 0 ) {
				curr = curr.succ;
			}
		}


		/**
		 * Inserts an object before the cursor
		 * @param o
		 */

		public void add(Object o) {
			curr.InsertBefore(new Node(o));
			size++;
			location++;
		}


		/**
		 * returns true if there are more elements in this list
		 */

		public boolean hasNext() {
			if( curr.succ != head ) {
				return true;
			}

			return false;
		}

		/**
		 * Returns true if the list have more elements when traversing backwards
		 * @return
		 */

		public boolean hasPrevious() {
			if( curr != head  ) {
				return true;
			}
			return false;
		}

		/**
		 * Gets the next element in the list and move forward
		 * @return
		 */

		public Object next() {
			if( curr.succ == head ) {
				throw new NoSuchElementException();
			}

			curr = curr.succ;
			location = (location+1)% size;
			return curr.data;
		}

		/**
		 * Returns the index of the element that would be returned by a next call
		 * @return
		 */

		public int nextIndex() {
			return location;
		}

		/**
		 * Return the previous element in the list and move backwards
		 * @return
		 */

		public Object previous() {
			if( curr == head ) {
				throw new NoSuchElementException();
			}

			curr = curr.pred;
			location = (location-1)% size;
			return curr.data;
		}

		/**
		 * Returns the index of the element that would be returned by a previous call
		 * @return
		 */

		public int previousIndex() {
			return location-1;
		}

		/**
		 * Removes the element returned from the list the element that was accessed by the last previous or next operation
		 */

		public void remove() {
			curr.Remove();
			size--;
			location--;
		}

		/**
		 * Sets the element returned from the list the element that was accessed by the last previous or next operation
		 * @param o
		 */

		public void set(Object o) {
			curr.data = o;
		}

	}
}
