/*
 * Date: Apr 23, 2010
 * Time: 12:23:56 PM
 * (c) 2010 Livescribe, Inc.
 */
package com.livescribe.ext.util;

/**
 * A latch that can be used for thread signalling.
 * <p>
 * Note: This uses the current instance for the wait/notify object.</p>
 *
 * @since 2.4
 *
 * @author Shawn Silverman
 */
public class Latch {
    private boolean state;

    /**
     * Creates a new latch.  The latch will not be set.
     */
    public Latch() {
        super();
    }

    /**
     * Sets the latch.  If the latch is not already set, then any threads
     * waiting for the latch to be set will be notified.  Once the latch is
     * set, it stays set until {@link #resetLatch()} is called.
     *
     * @see #resetLatch()
     */
    public synchronized void setLatch() {
        if (!state) {
            state = true;

            notifyAll();
        }
    }

    /**
     * Waits for the latch to be set.  This returns immediately if the latch
     * is already set.
     *
     * @throws InterruptedException if the thread was interrupted while
     *         waiting for the latch to be set.
     * @see #setLatch()
     */
    public synchronized void waitLatch() throws InterruptedException {
        while (!state) {
            wait();
        }
    }

    /**
     * Waits for the latch to be set.  This returns immediately if the latch
     * is already set.
     * <p>
     * This waits for the specified period of time.  If the time elapses
     * without the latch being set, then this returns <code>false</code>.
     * Otherwise, if the latch is set before the time elapses, then this
     * returns <code>true</code>.</p>
     * <p>
     * As in the behaviour for {@link Object#wait(long)}, a timeout of zero
     * indicates that we should wait forever until notified or interrupted.</p>
     *
     * @param timeout the maximum time to wait, in ms
     * @return whether the latch was set before the timeout elapsed.
     * @throws InterruptedException if the thread was interrupted while
     *         waiting for the latch to be set.
     * @throws IllegalArgumentException if the timeout value is negative.
     * @see #setLatch()
     */
    public synchronized boolean waitLatch(long timeout)
        throws InterruptedException
    {
        // Check the argument

        if (timeout < 0) {
            throw new IllegalArgumentException("Negative timeout: " + timeout);
        }

        // A zero timeout means wait forever

        if (timeout == 0) {
            waitLatch();

            return true;
        } else {
            long end = System.currentTimeMillis() + timeout;
            while (!state) {
                timeout = end - System.currentTimeMillis();
                if (timeout <= 0) {
                    break;
                }

                wait(timeout);
            }
            return state;
        }
    }

    /**
     * Checks if the latch is set.
     *
     * @return whether the latch is set.
     */
    public synchronized boolean isSet() {
        return state;
    }

    /**
     * Atomically sets the latch if it is not already set.  This returns
     * <code>true</code> if the latch state changed, and <code>false</code>
     * otherwise.
     *
     * @return whether the latch state was changed.
     */
    public synchronized boolean setIfNotSet() {
        if (!state) {
            state = true;
            notifyAll();

            return true;
        } else {
            return false;
        }
    }

    /**
     * Resets the latch.  The state is set to <code>false</code>.  No threads
     * are notified.
     *
     * @see #setLatch()
     */
    public synchronized void resetLatch() {
        state = false;
    }
}
