package com.livescribe.ext.ui;

import com.livescribe.display.Canvas;
import com.livescribe.display.Graphics;
import com.livescribe.display.Image;

import java.util.Vector;

/**
 * This class enables most types of displays that involves <em>N</em>
 * scrolling reels with a number of items in each reel.  Each reel can be in a
 * selected or unselected state, with a different image displayed for both.
 * <p>
 * A reel is like a virtual cylinder, where the last item wraps around to the
 * first (index zero).  It can be positioned exactly at an item, or in between
 * two items, by using a floating-point number.</p>
 * <p>
 * <img src="doc-files/SlotMachine-pic.png"/></p>
 *
 * @since 2.4
 * @author slynggaard
 */
// Docs by Shawn
public class SlotMachine extends Canvas {
    /**
     * Represents a single reel.
     */
    private final class Reel {
        Image[] unselectedImages;
        Image[] selectedImages;

        int maxWidth;
        int itemCount;

        // State

        float currPosition;
        boolean isSelected;

        /**
         * Creates a new slot reel object.  The normal and selected images
         * should be arrays having the same length.  The <code>selected</code>
         * parameter may be <code>null</code> if no selected images are
         * specified for this reel.
         * <p>
         * If the height of any selected images are different than the height
         * for the corresponding unselected images then the height of the
         * unselected images are used for any calculations.</p>
         *
         * @param unselected the unselected images
         * @param selected the selected images, may be <code>null</code>
         * @throws NullPointerException if <code>unselected</code> is
         *         <code>null</code>.
         * @throws IllegalArgumentException if <code>selected</code> is not
         *         <code>null</code> and its length does not equal the length
         *         of <code>unselected</code>.
         */
        Reel(Image[] unselected, Image[] selected) {
            if (selected != null && selected.length != unselected.length) {
                throw new IllegalArgumentException();
            }

            this.itemCount = unselected.length;

            this.unselectedImages = unselected;
            this.selectedImages   = selected;

            int maxWidth = 0;

            for (int i = unselected.length; --i >= 0; ) {
                if (unselected[i] != null && unselected[i].getWidth() > maxWidth) {
                    maxWidth = unselected[i].getWidth();
                }

                if (selected != null) {
                    if (selected[i] != null && selected[i].getWidth() > maxWidth) {
                        maxWidth = selected[i].getWidth();
                    }
                }
            }

            this.maxWidth = maxWidth;
        }

        /**
         * Paints this reel.  The vertical position, specified by
         * {@link #currPosition} can be any floating-point number, and will be
         * considered to wrap around the "cylinder".  For example, if image 1
         * should be displayed a further halfway up, specify the position as
         * 1.5.
         *
         * @param g the graphics context
         * @param x the X-coordinate
         * @return the width of this reel.
         */
        //* @param pos the vertical position of the reel on the cylinder
        //* @param selected whether to paint the selected version of the image
        int paint(Graphics g, int x) {
            int imageNum = (int)currPosition;
            float frac = currPosition - imageNum;
            imageNum = mod(imageNum, unselectedImages.length);

            int height = unselectedImages[imageNum].getHeight();
            int pixelShift = (int)(frac * height);
            int canvasHeight = g.getClipHeight();

            int y = ((canvasHeight - height) >> 1) - pixelShift;
            int y2 = y;
            int imageNum2 = imageNum;

            // If the image is not selected or not close enough to perfect
            // alignment then draw the normal image

            Image img = (selectedImages == null || !isSelected || pixelShift != 0)
                    ? unselectedImages[imageNum]
                    : selectedImages[imageNum];

            if (img != null) {
                g.drawImage(img, x, y);
            }

            // Draw images up until we reach the screen edge

            do {
                y -= unselectedImages[imageNum].getHeight();
                imageNum = mod(imageNum - 1, unselectedImages.length);
                g.drawImage(unselectedImages[imageNum], x, y);
            } while (y + unselectedImages[imageNum].getHeight() >= 0);

            // Draw images down until we reach the screen edge

            while (y2 < canvasHeight) {
                y2 += unselectedImages[imageNum2].getHeight();
                imageNum2 = mod(imageNum2 + 1, unselectedImages.length);
                g.drawImage(unselectedImages[imageNum2], x, y2);
            }

            return maxWidth;
        }
    }

    private int startX;

    private Vector reelsAndSeps;

    /** Maps a reel index into the proper index in {@link #reelsAndSeps}. */
    private Vector reelMap;

    /**
     * Creates a new slot machine UI having its drawing offset at zero.
     */
    public SlotMachine() {
        this(0);
    }

    /**
     * Creates a new slot machine UI having the specified drawing offset.
     *
     * @param startX start offset in pixels from the left side of the canvas
     */
    public SlotMachine(int startX) {
        this.startX = startX;
        this.reelsAndSeps = new Vector();
        this.reelMap = new Vector();
    }

    /*
     * Sets the images to use between the reels.  The array needs to contain
     * the {@linkplain #getReelCount()}+1 images.  The first and last will be
     * used for the left and right images, respectively, and the middle images
     * will be used as the separators.  The default is to have no separator
     * images.  This is the same as passing <code>null</code>.
     * <p>
     * If any of the images are <code>null</code> then they will be
     * substituted with zero-width images on the display.  As well, it is
     * suggested that the images be the same height as the display.</p>
     * <p>
     * The array is used directly and is not copied to an internal array.</p>
     *
     * @param images the array of images to be used as separators, may be
     *               <code>null</code>
     * @throws IllegalArgumentException if the array is not the proper length.
     *
    public void setSeparatorImages(Image[] images) {
        if (images != null && images.length != getReelCount() + 1) {
            throw new IllegalArgumentException();
        }

        separators = images;
    }*/

    /**
     * Adds a reel containing the specified images.  The number of items on
     * the reel will be equal to the number of images and its width will be
     * the maximum width of all the images.
     * <p>
     * There will be no selected images.</p>
     *
     * @param normalImages the array of unselected images
     * @see #addReel(Image[], Image[])
     */
    public void addReel(Image[] normalImages) {
        addReel(normalImages, null);
    }

    /**
     * Adds a reel containing the specified unselected and selected images.
     * The number of items on the reel will be equal to the number of images
     * and its width will be the maximum width of all the images.  The
     * selected images are alternates that will be used if the reel is not set
     * to a fractional position and if the reel is selected.
     *
     * @param unselected the array of normal images
     * @param selected the array of selected images, may be <code>null</code>
     * @throws NullPointerException if <code>normal</code> is
     *         <code>null</code>.
     * @throws IllegalArgumentException if the selected image array size is
     *         not equal to the normal image array size.
     */
    public void addReel(Image[] unselected, Image[] selected) {
        if (selected != null && selected.length != unselected.length) {
            throw new IllegalArgumentException();
        }

        reelMap.addElement(new Integer(reelsAndSeps.size()));  // NOTE This must be called before adding to 'reelsAndSeps'
        reelsAndSeps.addElement(new Reel(unselected, selected));
    }

    /**
     * Adds a separator image.
     *
     * @param separator the separator image
     */
    public void addSeparator(Image separator) {
        reelsAndSeps.addElement(separator);
    }

    /**
     * Gets the reel for the given reel index from {@link #reelsAndSeps}.
     * This first needs to consult with {@link #reelMap} to figure out where
     * the actual reel object is.
     *
     * @param n the reel number
     * @return the specified reel from {@link #reelsAndSeps}.
     * @throws IndexOutOfBoundsException if the reel number is out of range.
     */
    private Reel getReel(int n) {
        int index = ((Integer)reelMap.elementAt(n)).intValue();
        return (Reel)reelsAndSeps.elementAt(index);
    }

    /**
     * Gets the reel position as a floating-point value.  If the fraction part
     * is zero then the reel is exactly in the middle of one of its elements.
     * <p>
     * Note that the position will be in the range 0&ndash;<em>N</em>, not
     * including <em>N</em>, where <em>N</em> is the
     * {@linkplain #getReelCount() reel count}.</p>
     *
     * @param n the reel number
     * @return the reel location, a floating-point value.
     * @throws IndexOutOfBoundsException if the reel number is out of range.
     */
    public float getReelPosition(int n) {
        Reel reel = getReel(n);
        return mod(reel.currPosition, reel.itemCount);
    }

    /**
     * Gets the number of reels.
     *
     * @return the number of reels.
     * @see #addReel(Image[], Image[])
     */
    public int getReelCount() {
        return reelMap.size();
    }

    /**
     * Sets the location for the specified reel.  The location is specified as
     * a floating-point value, with non-integer numbers representing positions
     * between two items on the reel.  Its value starts at zero.
     * <p>
     * Note that the position properly wraps around the "cylinder".  It is not
     * necessary to worry about keeping track of numbers that are outside of
     * the range 0&ndash;<em>N</em> (where <em>N</em> is the
     * {@linkplain #getReelCount() reel count}.</p>
     * <p>
     * For example, a position of 2 will show image 2 of reel <code>n</code>.
     * A position of 2.4 will show the same image, but scrolled up a farther
     * 40% of its height.</p>
     *
     * @param n the reel number
     * @param position the new position, a floating-point value
     * @throws IndexOutOfBoundsException if the reel number is out of range.
     * @see #getReelCount()
     */
    public void setReelPosition(int n, float position) {
        getReel(n).currPosition = position;
    }

    /**
     * Moves the reel a specified amount.  A positive value moves the image up
     * and a negative value moves the image down.
     *
     * @param n the reel number
     * @param delta move the reel by this amount
     * @throws IndexOutOfBoundsException if the reel number is out of range.
     * @see #getReelCount()
     * @see #setReelPosition(int, float)
     */
    public void moveReel(int n, float delta) {
        getReel(n).currPosition += delta;
    }

    /**
     * Selects the specified reel.  Note that this does not unselect the other
     * reels.
     *
     * @param n the reel number
     * @param flag indicates whether to select the reel
     * @throws IndexOutOfBoundsException if the reel number is out of range.
     * @see #getReelCount()
     */
    public void selectReel(int n, boolean flag) {
        getReel(n).isSelected = flag;
    }

    /**
     * Paints this UI component.
     *
     * @param g the graphics context
     */
    protected void paint(Graphics g) {
        int currX = startX;
        int count = reelsAndSeps.size();

        for (int i = 0 ; i < count ; i++) {
            Object o = reelsAndSeps.elementAt(i);

            if (o instanceof Reel) {
                currX += ((Reel)o).paint(g, currX);
            } else if (o instanceof Image) {
                Image img = (Image)o;
                g.drawImage(img, currX, 0);
                currX += img.getWidth();
            }
        }
    }

    /**
     * Finds the modulus of two numbers.  For example, if "mod&nbsp;3" were
     * used, then the numbers would include:
     * {&hellip;2,0,1,2,<strong>0</strong>,1,2,0,&hellip;}.
     *
     * @param value the first value
     * @param mod the modulus
     * @return the result.
     */
    private static int mod(int value, int mod) {
        value = value % mod;
        if (value < 0) {
            value += mod;
        }
        return value;
    }

    /**
     * Finds the modulus of two numbers.  For example, if "mod&nbsp;3" were
     * used, then the numbers would include:
     * {&hellip;2,0,1,2,<strong>0</strong>,1,2,0,&hellip;}.
     *
     * @param value the first value
     * @param mod the modulus
     * @return the result.
     */
    private static float mod(float value, int mod) {
        value = value % mod;
        if (value < 0) {
            value += mod;
        }
        return value;
    }
}
