/*
 * Date: 29-Nov-2009
 * Time: 10:18:27 AM
 * (c) 2010 Livescribe, Inc.
 */
package com.livescribe.ext.ui;

import com.livescribe.display.*;
import com.livescribe.ui.ScrollLabel;

import java.io.InputStream;
import java.io.IOException;
import java.util.Vector;

/**
 * A menu where only one item can be selected at once.  Selected and
 * not-selected items have an associated icon.
 *
 * @since 2.3
 * @author Shawn Silverman
 */
public class RadioMenu extends Menu {
    // DONETODO System resources
    //          Done with some settings in this class
    //private static final String DOT_IMAGE       = "/images/radio-dot.arw";
    //private static final String EMPTY_DOT_IMAGE = "/images/empty-dot.arw";

    // Default Images

    private static String defaultSelectedImage;
    private static String defaultNotSelectedImage;

    // Instance Images

    private String selectedImage;
    private String notSelectedImage;

    private Image dotImage;  // The cached selected image
    private Image emptyDotImage;  // The cached not selected image

    private Vector radioIcons;

    // TODO Potentially add this later
    //private MediaPlayer player;
    private Display display;

    private int selectedIndex;

    // Properties

    /**
     * Indicates whether to jump to the currently selected image when the
     * menu is activated.
     */
    private boolean jumpToSelected = true;

    /**
     * Listens to the selection and sets the appropriate images.
     */
    private final class MyListener implements Menu.MenuListener {
        public void menuShown(Menu menu) {
        }

        public void itemChanged(Menu menu, int oldIndex, int newIndex, boolean menuActivated) {
            // Change the focus to the selected item if the menu was just
            // activated

            if (menuActivated && jumpToSelected) {
                if (menu.getFocusIndex() != selectedIndex) {
                    menu.setFocusIndex(selectedIndex);
                }
            }
        }

        public void itemSelected(Menu menu, int index) {
            if (selectedIndex != index) {
                setSelectedIndex(index);

                // Re-show the browse list so that the image update gets picked up

                display.setTransition(Transition.DEFAULT);//RIGHT_TO_LEFT);
                display.setCurrent(getBrowseList());
            } else {
                // No change

                // TODO Potentially add this later
                //if (player != null) player.play(MENU_SELECT_SOUND);
            }
        }
    }

    /**
     * This browse list implementation handles painting icons properly.
     */
    private final class RadioMenuBrowseList extends MenuBrowseList {
        /**
         * Creates a new browse list.
         *
         * @param items the items
         * @param title the title
         */
        RadioMenuBrowseList(Vector items, Title title) {
            super(items, title);
        }

        // TODO Request the item index!!
        protected void drawItem(Graphics g, Item item,
                                Image icon, Object text, int x, int y) {
            // First, find the index of the item

            int index = ((MenuItem)item).getIndex();

            // Draw the radio icon

            if (0 <= index && index < radioIcons.size()) {
                Image radioIcon = (Image)radioIcons.elementAt(index);
                if (radioIcon != null) {
                    g.drawImage(radioIcon, x, y);

                    // Allow 2 pixels of spacing

                    x += radioIcon.getWidth() + 1;
                }
            }

            // Draw the icon

            if (icon != null) {
               g.drawImage(icon, x, y);

               // Allow 2 pixels of spacing

               x += icon.getWidth() + 1;
            }

            // Draw the text

            if (text instanceof String) {
                g.drawString((String)text, x, y, 0);
            } else if (text instanceof AttributedText) {
                g.drawString((AttributedText)text, x, y, 0);
            }
        }

        protected void scrollItem(Graphics g, ScrollLabel scroller, Item item, int x, int y) {
            Image icon = item.getIcon();

            // COPIED from superclass
            // we need to draw the icon static and allow the text to scroll, so
            // use the BrowseList's graphics to render the image, then adjust
            // scroller's clip so that we can just scroll the text.

            // First, find the index of the item

            int i = itemModel.indexOf(item);

            // Draw the radio icon

            if (0 <= i && i < radioIcons.size()) {
                Image radioIcon = (Image)radioIcons.elementAt(i);
                if (radioIcon != null) {
                    g.drawImage(radioIcon, x, y);

                    // Allow 2 pixels of spacing

                    x += radioIcon.getWidth() + 1;
                }
            }

            // Draw the icon

            if (icon != null) {
               g.drawImage(icon, x, y);

               // Allow 2 pixels of spacing

               x += icon.getWidth() + 1;
            }

            // Draw the text

            Object text = item.getText();

            short scrollerWidth = (short)(g.getClipWidth() - x - insets.right);
            int slidePanelHeight = Display.getHeight() - insets.top - insets.bottom;

            setScrollerClipRegion(scroller, (short)x, (short)(insets.top), scrollerWidth, (short)slidePanelHeight);

            if (text instanceof String) {
                scroller.draw(null, (String)text, x, y, scrollerWidth, slidePanelHeight, true);
            } else if (text instanceof AttributedText) {
                scroller.draw((AttributedText)text, null, x, y, scrollerWidth, slidePanelHeight, true);
            }
        }
    }

    /**
     * A convenience method that creates a radio menu containing "Off" and
     * "On".
     *
     * @param flag the initial selection, <code>true</code> for "On" and
     *             <code>false</code> for "Off"
     * @param display the display object
     * @return a new On/Off radio menu.
     */
    public static Menu createOnOffMenu(boolean flag, Display display) {
//* @param player a {@link MediaPlayer}
        // TODO Internationalize this
        return new RadioMenu(
                new String[] { "Off", "On" }, flag ? 1 : 0,
                display);
    }

    /**
     * Creates a new menu with the specified default selection.
     *
     * @param text the menu items
     * @param selectedIndex the initial selection
     * @param display the display object
     * @throws IndexOutOfBoundsException if the selected index is out of range.
     */
    public RadioMenu(String[] text, int selectedIndex, Display display) {
//* @param player a {@link MediaPlayer}
        if (selectedIndex < 0 || text.length <= selectedIndex) {
            throw new IndexOutOfBoundsException("Selected index out of range: " + selectedIndex);
        }

        // TODO Potentially add this later
        //this.player = player;
        this.display = display;

        addMenuListener(new MyListener());

        radioIcons = new Vector(text.length);
        radioIcons.setSize(text.length);

        for (int i = 0, len = text.length; i < len; i++) {
            add(text[i]);
            setRadioIcon(i, loadImage(i == selectedIndex));
        }

        this.selectedIndex = selectedIndex;

        // Select the proper item

        getBrowseList().setFocusItem(selectedIndex);
    }

    /**
     * Creates our own implementation of a browse list.
     *
     * @param items the items
     * @param title the title
     * @return a new browse list implementation.
     */
    protected MenuBrowseList createBrowseList(Vector items, BrowseList.Title title) {
        return new RadioMenuBrowseList(items, title);
    }

    /**
     * Sets the default image for a selected item.  This is set to
     * <code>null</code> by default.
     *
     * @param image the new default <em>selected</em> image, <code>null</code>
     *             to unset
     */
    public static void setDefaultSelectedImage(String image) {
        defaultSelectedImage = image;
    }

    /**
     * Sets the default image for an item that is not selected.  This is set
     * to <code>null</code> by default.
     *
     * @param image the new default <em>not selected</em> image,
     *             <code>null</code> to unset
     */
    public static void setDefaultNotSelectedImage(String image) {
        defaultNotSelectedImage = image;
    }

    /**
     * Sets the image to use for a selected item.  This overrides the
     * {@link #setDefaultSelectedImage(String) default image}.  If this is set
     * to <code>null</code> then the default image will be used.  This is set
     * to <code>null</code> by default.
     *
     * @param image the new <em>selected</em> image, <code>null</code> to unset
     * @see #setDefaultSelectedImage(String)
     */
    public synchronized void setSelectedImage(String image) {
        this.selectedImage = image;

        this.dotImage = null;
    }

    /**
     * Sets the image to use for an item that is not selected.  This overrides
     * the {@link #setDefaultNotSelectedImage(String) default image}.  If this
     * is set to <code>null</code> then the default image will be used.  This
     * is set to <code>null</code> by default.
     *
     * @param image the new <em>not selected</em> image, <code>null</code> to
     *              unset
     * @see #setDefaultNotSelectedImage(String)
     */
    public synchronized void setNotSelectedImage(String image) {
        this.notSelectedImage = image;

        this.emptyDotImage = null;
    }

    /**
     * Gets the index of the currently selected item.
     *
     * @return the selected item index.
     */
    public int getSelectedIndex() {
        return selectedIndex;
    }

    /**
     * Indicates whether the menu should jump to the currently selected item
     * when it is activated.  The default is <code>true</code>.
     *
     * @param flag the new state
     */
    public void setJumpToSelected(boolean flag) {
        this.jumpToSelected = flag;
    }

    /**
     * Sets the icon for the item at the specified index.
     *
     * @param index set the icon for the item at this index
     * @param icon the new icon, may be <code>null</code>
     * @throws IndexOutOfBoundsException if the index is out of range.
     * @see #getItemCount()
     */
    private void setRadioIcon(int index, Image icon) {
        if (index < 0 || getItemCount() <= index) {
            throw new IndexOutOfBoundsException("Index out of range: " + index);
        }

        radioIcons.setElementAt(icon, index);
    }

    /**
     * Sets the index of the currently selected item.  This also swaps the
     * images, so be sure to redisplay the browse list.
     *
     * @param index the new selected item index
     * @throws IndexOutOfBoundsException if the index is out of range.
     * @see #getItemCount()
     */
    public void setSelectedIndex(int index) {
        if (index < 0 || getItemCount() <= index) {
            throw new IndexOutOfBoundsException("Index out of range: " + index);
        }

        if (index != selectedIndex) {
            setRadioIcon(selectedIndex, loadImage(false));
            this.selectedIndex = index;
            setRadioIcon(index, loadImage(true));
        }
    }

    /**
     * Loads the specified image.
     *
     * @param selected whether to load the selected or not-selected image
     * @return the loaded image, or <code>null</code> if the image could not
     *         be loaded.
     */
    private synchronized Image loadImage(boolean selected) {
        String imgSrc;

        if (selected) {
            if (dotImage != null) return dotImage;

            imgSrc = (selectedImage == null) ? defaultSelectedImage : selectedImage;
        } else {
            if (emptyDotImage != null) return emptyDotImage;

            imgSrc = (notSelectedImage == null) ? defaultNotSelectedImage : notSelectedImage;
        }

        // Load the image

        InputStream in = getClass().getResourceAsStream(imgSrc);

        if (in != null) {
            try {
                Image img = Image.createImage(in);
                if (selected) {
                    dotImage = img;
                } else {
                    emptyDotImage = img;
                }

                return img;
            } catch (IOException ex) {
                ex.printStackTrace();
            } catch (IllegalArgumentException ex) {
                ex.printStackTrace();
            } finally {
                try { in.close(); } catch (IOException ex) { ex.printStackTrace(); }
            }
        }

        return null;
    }
}
