/*
 * Date: 26-Mar-2010
 * Time: 1:06:34 AM
 * (c) 2010 Livescribe, Inc.
 */
package com.livescribe.ext.ui;

import com.livescribe.buttons.Bookmarkable;
import com.livescribe.display.Transition;
import com.livescribe.event.*;
import com.livescribe.penlet.AbstractPenletPlugin;
import com.livescribe.penlet.Penlet;
import com.livescribe.penlet.PenletContext;
import com.livescribe.penlet.PenletPlugin;
import com.livescribe.ui.MediaPlayer;

import java.util.Hashtable;

/**
 * A plugin that handles menu bookmark activation.  This does not consume any
 * events.
 * <p>
 * When the user taps on a bookmarked region, the following things happen:</p>
 * <ol>
 * <li>The <em>select</em> sound is played if set,</li>
 * <li>The bookmarked menu item is shown, and</li>
 * <li>If the bookmark was configured to be <em>selected</em> when activated,
 *     then either the <em>forward</em> or <em>select</em> sound is played,
 *     depending on whether the menu has a submenu or not.</li>
 * </ol>
 * <p>
 * A usage example:</p>
 * <blockquote>
 * <pre>MenuBookmarkHandlerPlugin mbh = new MenuBookmarkHandlerPlugin(this);
context.installPlugin(mbh);
mbh.activate();
&hellip;
myMenu.add("Info");
mbh.registerLast(myMenu, new BookmarkInfo(4, "INFO"));
myMenu.add("About");
mbh.registerLast(myMenu, new BookmarkInfo(5, "ABOUT"));</pre>
 * </blockquote>
 *
 * @see #setForwardAPM(String)
 * @see #setSelectAPM(String)
 * @see PenletContext#installPlugin(PenletPlugin)
 * @since 2.3
 *
 * @author Shawn Silverman
 */
/* OLD COMMENTS:
 * Handles menu bookmark details so that the application does not have to.
 * Specifically, this handles the pen entering a region and a menu item needs
 * to be displayed.
 */
public class MenuBookmarkHandlerPlugin extends AbstractPenletPlugin {
    private Hashtable idMap = new Hashtable();

    private long parentHoldTime = 1000L;

    private String selectAPM;
    private String forwardAPM;

    private MediaPlayer player;  // Set to non-null when one of the APM's is set to non-null

    /**
     * Keeps track of a menu and an associated item index.
     */
    private static final class MenuItemInfo {
        Menu menu;
        int index;

        boolean doSelect;

        /**
         * Creates a new menu item info object.
         *
         * @param menu the menu
         * @param index the associated item index
         * @param doSelect whether to <em>select</em> the item when it's
         *                 activated
         */
        MenuItemInfo(Menu menu, int index, boolean doSelect) {
            this.menu  = menu;
            this.index = index;

            this.doSelect = doSelect;
        }
    }

    /**
     * Creates a new handler for activating menu bookmarks.
     *
     * @param penlet the associated penlet
     */
    public MenuBookmarkHandlerPlugin(Penlet penlet) {
        super(penlet, false);  // Don't consume all events
    }

    /**
     * Registers a menu item with this plugin so that when it is activated
     * with a button tap, it is displayed properly.  This also sets the item
     * as bookmarkable.
     * <p>
     * If the <code>doSelect</code> parameter is <code>true</code>, then the
     * menu item will be <em>selected</em> when it is activated.  This means
     * that it will be as if the user tapped <em>right</em> on the NavPlus
     * when the menu is showing.  It also means that if the item has a submenu
     * then it will be followed after showing the item for a small period of
     * time.</p>
     * <p>
     * This makes use of the {@link Menu#setBookmarkable(int, Bookmarkable)}
     * method so it is not necessary to call this yourself.</p>
     *
     * @param menu register this menu
     * @param index the menu item index
     * @param b the {@link Bookmarkable} information
     * @param doSelect whether to <em>select</em> the item when it's activated
     * @see Menu#setBookmarkable(int, Bookmarkable)
     */
    public void register(Menu menu, int index, Bookmarkable b, boolean doSelect) {
        // Set the item as bookmarkable

        menu.setBookmarkable(index, b);

        // Map the ID

        if (b != null) {
            idMap.put(
                    new Integer(b.getBookmarkAreaId()),
                    new MenuItemInfo(menu, index, doSelect));
        }
    }

    /**
     * Registers the last item in the menu.  Please see
     * {@link #register(Menu, int, Bookmarkable, boolean)} for more details.
     *
     * @param menu register the last item in this menu
     * @param b the {@link Bookmarkable} information
     * @param doSelect whether to <em>select</em> the item when it's activated
     * @see #register(Menu, int, Bookmarkable, boolean)
     */
    public void registerLast(Menu menu, Bookmarkable b, boolean doSelect) {
        register(menu, menu.getItemCount() - 1, b, doSelect);
    }

    /*
     * Sets whether to <em>select</em> the menu when activating it.  This is
     * equivalent to the user tapping <em>right</em> on the NavPlus.  For
     * example, if the menu has a submenu, then the submenu will be followed.
     * Additionally, if there is a submenu, then the menu&dash;the parent, in
     * this case&mdash;will be shown for a small amount of time and then the
     * submenu will be followed.
     * <p>
     * The default is to <em>select</em> the menu.</p>
     *
     * @param flag whether to select the activated menus
     * @see #setParentHoldTime(long)
     *
    /* Sets whether to follow submenus when a menu having a submenu is
     * bookmarked and the pen taps in the bookmarked area on the page.*
    public void setSelectOnActivate(boolean flag) {
        selectOnActivate = flag;
    }*/

    /**
     * Sets the amount of time to pause at the parent menu when following into
     * submenus.  This option is ignored if the item is configured to not be
     * <em>selected</em> when activated. The value is set to zero if the
     * parameter is &le; zero.
     * <p>
     * Note that if there is no submenu then there is no pause.</p>
     * <p>
     * The default is 1000&nbsp;ms.</p>
     *
     * @param time the pause time, in milliseconds
     */
    public void setParentHoldTime(long time) {
        if (time < 0) time = 0L;

        parentHoldTime = time;
    }

    /**
     * Sets the sound to play when a menu item is <em>selected</em>.  This is
     * set to <code>null</code> by default.
     * <p>
     * This sound will play instead of any sound configured on the menu, even
     * if this is set to <code>null</code>.</p>
     *
     * @param clip the new <em>select</em> sound clip, <code>null</code> to
     *             unset
     * @see Menu#setSelectAPM(String)
     */
    public void setSelectAPM(String clip) {
        selectAPM = clip;

        // Set the media player

        if (clip != null && player == null) {
            player = MediaPlayer.newInstance(getPenlet());
        }
    }

    /**
     * Sets the sound to play when a menu item's submenu is <em>followed</em>.
     * This is set to <code>null</code> by default.
     * <p>
     * This sound will play instead of any sound configured on the menu, even
     * if this is set to <code>null</code>.</p>
     *
     * @param clip the new <em>follow</em> sound clip, <code>null</code> to
     *             unset
     * @see Menu#setForwardAPM(String)
     */
    public void setForwardAPM(String clip) {
        forwardAPM = clip;

        // Set the media player

        if (clip != null && player == null) {
            player = MediaPlayer.newInstance(getPenlet());
        }
    }

    /**
     * This processes pen-down events but does not consume them.  They are
     * still passed on to the application since it may still wish to be
     * informed of the button taps.
     * <p>
     * Menus are selected before the event is passed to the application.</p>
     *
     * @param ev the pen tip event
     * @return <code>false</code> indicating that the events will be passed
     *         to the application.
     */
    public boolean onPenTipEvent(PenTipEvent ev) {
        switch (ev.getId()) {
            case PenTipEvent.PEN_TIP_PEN_DOWN:
                // Activate the associated menu item

                MenuItemInfo item = (MenuItemInfo)idMap.get(
                        new Integer(ev.getRegion().getAreaId()));

                if (item != null) {
                    // First, play the "select" sound

                    if (selectAPM != null) {
                        if (player != null) player.play(selectAPM);
                    }

                    // Show the bookmarked menu

                    item.menu.setFocusIndex(item.index);
                    item.menu.show(getDisplay(), Transition.DEFAULT);

                    // Possibly delay and show the submenu

                    if (item.doSelect) {
                        Menu submenu = item.menu.getSubmenu(item.index);
                        if (submenu != null) {
                            // Check if we need to delay

                            if (parentHoldTime > 0L) {
                                try {
                                    Thread.sleep(parentHoldTime);
                                } catch (InterruptedException ex) {
                                    // Somebody doesn't want us to continue

                                    return false;
                                }
                            }

                            if (forwardAPM != null) {
                                if (player != null) player.play(forwardAPM);
                            }
                        } else {
                            // Don't play the select sound because we've just played it

                            //if (selectAPM != null) {
                            //    player.play(selectAPM);
                            //}
                        }

                        // Actually show the submenu

                        item.menu.doSelect(item.index, getDisplay(), null);
                    }
                }
                break;
        }

        return false;
    }
}
