/*
 * Date: Oct 15, 2009
 * Time: 10:47:11 AM
 * (c) 2010 Livescribe, Inc.
 */
package com.livescribe.ext.ui;

import com.livescribe.display.*;
import com.livescribe.event.MenuEvent;
import com.livescribe.ui.MediaPlayer;
import com.livescribe.buttons.Bookmarkable;
import com.livescribe.buttons.BookmarkInfo;

import java.util.Vector;
import java.io.InputStream;

/**
 * A class to make it easier to build browse list-based menus.  This
 * represents a list of items, some of which may be selectable submenus.
 * <p>
 * Internally, this maps browse list items to sub-menus.</p>
 *
 * @note This class is not thread safe.
 * @since 2.3
 *
 * @author Shawn Silverman
 */
//@SuppressWarnings("unchecked")
public class Menu {
    // TODO Reconsider these when an app can use system resources; in the meantime, don't disable default left and right APM's
    ///** Played when you can't go further into the menu, in any direction. */
    //private static final String MENU_TOP_SOUND     = "/audio/SL_NA.wav";

    ///** Played when jumping back to a parent menu. */
    //private static final String MENU_BACK_SOUND    = "/audio/SL_Back.wav";

    ///** Played when jumping forward to a submenu. */
    //private static final String MENU_FORWARD_SOUND = "/audio/SL_Forward.wav";

    ///** Played when selecting something. */
    //private static final String MENU_SELECT_SOUND = "/audio/SL_Select.wav";

    /** The internal browse list.  This is composition and not inheritance. */
    private BrowseList browseList;

    /** Must be kept in sync with the browse list's focus index. */
    private int focusIndex;

    /**
     * Indicates whether to notify listeners that an item has changed.  This
     * variable is necessary to avoid notifying twice when a browse list is
     * shown with {@link #show(Display, Transition)}.
     */
    private boolean dontNotifyItemChanged;

    /** A list of the menu items. */
    private Vector items;

    /** The menu title, can be {@link AttributedText} or a plain string. */
    private Object title;

    /** Listens to menu events. */
    private MenuListener listener;

    /** The parent menu, may be <code>null</code>. */
    private Menu parent;

    /**
     * Any metadata associated with the menu.  This may help with menu
     * identification purposes.
     */
    private Object metadata;

    // Sounds

    private static String defaultBackAPM;
    private static String defaultForwardAPM;
    private static String defaultSelectAPM;
    private String backAPM;
    private String forwardAPM;
    private String selectAPM;

    private boolean playBackAPM    = true;
    private boolean playForwardAPM = true;
    private boolean playSelectAPM  = true;

    /**
     * A special {@link BrowseList} implementation that allows the menu to
     * access some protected methods and override others.  This is necessary
     * to provide proper <em>item-changed</em> event notifications.
     * <p>
     * All items in this browse list will be an instance of
     * {@link MenuItem}.</p>
     *
     * @see MenuItem
     */
    protected class MenuBrowseList extends BrowseList {
        /**
         * Creates a new browse list.
         *
         * @param model the model
         * @param title the title
         */
        protected MenuBrowseList(Vector model, Title title) {
            super(model, title);
        }

        /**
         * This is overridden to intercept Flick n Scrub and other changes.
         * Subclasses overriding this method should call
         * <code>super.setFocusItem</code>.
         *
         * @param index the item index
         */
        public void setFocusItem(int index) {
            int oldIndex = Menu.this.focusIndex;
            super.setFocusItem(index);

            Menu.this.focusIndex = index;

            if (!dontNotifyItemChanged) {
                fireItemChanged(oldIndex, index, false);
            }
        }
    }

    /**
     * This interface provides a way for menu selection to trigger an action.
     * The action is executed when a menu item is selected by the NavPlus
     * <em>forward</em> arrow.  Note that the action will be executed whether
     * there is a submenu or not, and before any new menu is shown.
     * <p>
     * This provides an alternative to {@link MenuListener#itemSelected(Menu, int)}
     * by allowing an action to be tied to a specific menu item as opposed to
     * the whole menu.</p>
     *
     * @see MenuListener#itemSelected(Menu, int)
     *
     * @author Shawn Silverman
     */
    public static interface MenuAction {
        /**
         * The specified menu and menu selection index was just selected.
         *
         * @param menu the menu
         * @param index the menu's selection index
         */
        public void execute(Menu menu, int index);
    }

    /**
     * This interface listens to menu changes or events.
     *
     * @author Shawn Silverman
     */
    public static interface MenuListener {
        /*
         * The item in the specified menu was activated.  This is called when
         * the menu has newly become active.
         * <p>
         * This is the opposite of {@linkplain #itemSelected item selection}
         * in that an activated menu has newly become visible.  A selected
         * menu may lose visibility.</p>
         *
         * @param menu the menu
         * @param index the index of the newly shown item
         *
        public void itemActivated(Menu menu, int index);*/

        /**
         * The item in the specified menu was changed to the item having the
         * given index.  This is called when a list scrolls up or down or when
         * a new menu was made active.  The <code>menuActivated</code>
         * parameter discerns whether the menu was freshly activated or simply
         * changing items.
         * <p>
         * The new index can be -1 if the title is showing.</p>
         * <p>
         * Note: It is possible for the new index to equal the old index, for
         * example when the menu has been activated or when a focus change has
         * been requested and the item is not changed.</p>
         * <p>
         * Note 2: This can also be called when the index has changed without
         * the menu showing on the display.</p>
         *
         * @param menu the menu
         * @param oldIndex the index of the previously shown item
         * @param newIndex the index of the newly shown item
         * @param menuActivated indicates whether this menu was newly activated
         */
        public void itemChanged(Menu menu,
                                int oldIndex, int newIndex,
                                boolean menuActivated);
        // DONETODO Catch when the menu is first shown

        /*
         * The item in the specified menu is about to be changed to the item
         * having the given index.  This is called before the menu is shown,
         * and otherwise has the same behavior as
         * {@link #itemChanged(Menu, int, int, boolean)}.
         * <p>
         * This event is useful when state needs to be changed before the menu
         * is made visible or before items are actually changed.
         *
         * @param menu the menu
         * @param oldIndex the index of the previously shown item
         * @param newIndex the index of the newly shown item
         * @param menuActivated indicates whether this menu will be newly
         *                      activated
         *
        public void itemChanging(Menu menu,
                                 int oldIndex, int newIndex,
                                 boolean menuActivated);*/

        /**
         * An item in the specified menu was selected (activated, executed).
         * This is called when the user selects <em>forward</em> on a specific
         * menu item.  This is called regardless of whether the item is
         * {@linkplain Menu#isSelectable(int) selectable}.
         * <p>
         * This method provides a way for menu selection to trigger an action.
         * This will be called whether there is a submenu or not, and before
         * any new menu is shown.</p>
         * <p>
         * This provides an alternative to {@link MenuAction} which can be
         * tied to specific menu items rather than the whole menu.</p>
         *
         * @param menu the menu
         * @param index the selected item index
         * @see MenuAction
         */
        /*
         * <p>
         * This is the opposite of {@linkplain #itemActivated item activation}
         * in that a selected menu may lose its visibility.  An activated menu
         * gains visibility.</p>*/
        public void itemSelected(Menu menu, int index);

        /**
         * The specified menu was just shown on the display.
         *
         * @param menu the menu
         * @see Menu#show(Display, Transition)
         */
        public void menuShown(Menu menu);
    }

    /**
     * An implementation of a {@linkplain com.livescribe.display.BrowseList.Item browse list item}.
     * The items in the browse list will be an instance of this class.
     *
     * @see com.livescribe.display.BrowseList.Item
     */
    protected static class MenuItem implements BrowseList.Item {
        int index;

        Object text;
        Image icon;
        boolean selectable;

        MenuAction action;
        Menu submenu;

        String audio;

        /** Metadata. */
        Object metadata;

        /**
         * Creates a new item with the specified text, selectable state, and
         * other parameters.
         *
         * @param index the item's index number
         * @param text the item text
         * @param icon the item icon, may be <code>null</code>
         * @param audio the audio that should play when this item gets the
         *              focus
         * @param selectable whether the item is selectable
         * @param action the action to execute when selected, may be <code>null</code>
         * @param submenu the submenu, may be <code>null</code>
         * @param metadata any metadata
         */
        MenuItem(int index,
                 Object text, Image icon, String audio, boolean selectable,
                 MenuAction action, Menu submenu,
                 Object metadata) {
            this.index = index;

            // BrowseList things

            this.text = text;
            this.icon = icon;
            this.selectable = selectable;

            this.audio = audio;

            // Our menu things

            this.action = action;
            this.submenu = submenu;

            this.metadata = metadata;
        }

        /**
         * Creates a new menu item using the given object.
         *
         * @param index the item's index number
         * @param item the bookmarkable menu item
         */
        MenuItem(int index, BookmarkableMenuItem item) {
            this(index,
                    item.text, item.icon, item.audio, item.selectable,
                    item.action, item.submenu,
                    item.metadata);
        }

        public boolean isSelectable() { return selectable; }
        public Object getText() { return text; }

        public Image getIcon() { return icon; }

        public InputStream getAudioStream() {
            if (audio == null) return null;  // The following will throw an NPE if audio is null
            return getClass().getResourceAsStream(audio);
        }

        public String getAudioMimeType() {
            return MediaPlayer.getMimeType(audio);
        }

        /**
         * Gets the item's index number.
         *
         * @return the index number
         */
        protected int getIndex() {
            return index;
        }
    }

    /**
     * A bookmarkable browse list item.  This uses composition to implement
     * the bookmarking feature.
     */
    private static final class BookmarkableMenuItem extends MenuItem
        implements Bookmarkable
    {
        private Bookmarkable b;

        /**
         * Creates a new bookmarkable menu item.  This uses composition to
         * implement this feature.
         *
         * @param item wrap this menu item
         * @param b the bookmarkable item
         * @throws NullPointerException if any of the parameters are
         *         <code>null</code>.
         */
        BookmarkableMenuItem(MenuItem item, Bookmarkable b) {
            super(item.index,
                    item.text, item.icon, item.audio, item.selectable,
                    item.action, item.submenu,
                    item.metadata);

            if (b == null) throw new NullPointerException();

            this.b = b;
        }

        // Bookmarkable

        public int getBookmarkAreaId() {
            return b.getBookmarkAreaId();
        }

        public int getBookmarkZOrder() {
            return 0;
        }

        public String getBookmarkDisplayName() {
            return b.getBookmarkDisplayName();
        }

        /**
         * This returns <code>null</code> for the default displayable.
         *
         * @return <code>null</code>.
         */
        public Displayable getBookmarkDisplayable() {
            return b.getBookmarkDisplayable();
        }
    }

    /**
     * Creates a new menu with no title.
     */
    public Menu() {
        this(null);
    }

    /**
     * Creates a new menu having the specified title.    The title text may be
     * an {@link AttributedText} object or a plain string; it may also be
     * <code>null</code>.
     *
     * @param title the menu title, may be <code>null</code>
     * @throws IllegalArgumentException if the title is not valid text.
     */
    public Menu(Object title) {
        checkText(title);

        this.items = new Vector();
        this.title = title;

        focusIndex = (title != null) ? -1 : 0;
    }

    /**
     * Checks that the specified text is <code>null</code> or an instance of
     * {@link AttributedText} or {@link String}.  If not then this throws an
     * exception.
     *
     * @param text check this object
     * @throws IllegalArgumentException if the text fails the check.
     */
    private static void checkText(Object text) {
        if (!(text == null || text instanceof AttributedText || text instanceof String)) {
            throw new IllegalArgumentException("Bad text type: " + text.getClass().getName());
        }
    }

    /**
     * A listener that chains to another.  This is useful for looking at who's
     * in the chain, for example, to remove a listener.
     */
    private static final class ChainedMenuListener implements MenuListener {
        MenuListener ml1;
        MenuListener ml2;

        /**
         * Creates a new chained menu listener.
         *
         * @param ml1 the previous in the chain
         * @param ml2 the next in the chain
         */
        ChainedMenuListener(MenuListener ml1, MenuListener ml2) {
            this.ml1 = ml1;
            this.ml2 = ml2;
        }

        public void itemChanged(Menu menu, int oldIndex, int newIndex, boolean menuActivated) {
            ml1.itemChanged(menu, oldIndex, newIndex, menuActivated);
            ml2.itemChanged(menu, oldIndex, newIndex, menuActivated);
        }

        public void itemSelected(Menu menu, int index) {
            ml1.itemSelected(menu, index);
            ml2.itemSelected(menu, index);
        }

        public void menuShown(Menu menu) {
            ml1.menuShown(menu);
            ml2.menuShown(menu);
        }
    }

    /**
     * Adds a menu listener to this menu.  This does nothing if the listener
     * is <code>null</code>.
     *
     * @param listener the new menu listener
     */
    public void addMenuListener(final MenuListener listener) {
        // Chain the listeners

        if (this.listener == null) {
            this.listener = listener;
        } else if (listener != null) {
            // NOTE this.listener is not null

            this.listener = new ChainedMenuListener(this.listener, listener);
        }
    }

    /**
     * Removes a menu listener.  This does nothing if the listener is
     * <code>null</code> or if it has not already been added.
     *
     * @param listener the listener to remove
     * @since 2.5
     */
    public void removeMenuListener(MenuListener listener) {
        // Walk through the chain

        if (this.listener != null) {
            if (this.listener == listener) {
                this.listener = null;
            } else {
                MenuListener ml = this.listener;

                // The following algorithm walks backwards through the chain
                // so that the last added is the first removed

                ChainedMenuListener prev = null;
                while (ml instanceof ChainedMenuListener) {
                    ChainedMenuListener cml = (ChainedMenuListener)ml;
                    if (cml.ml2 == listener) {
                        if (prev != null) {
                            prev.ml1 = cml.ml1;
                        } else {
                            this.listener = cml.ml1;
                        }
                        return;
                    } else if (cml.ml1 == listener) {
                        if (prev != null) {
                            prev.ml1 = cml.ml2;
                        } else {
                            this.listener = cml.ml2;
                        }
                        return;
                    } else {
                        // Take advantage of the fact that a
                        // ChainedMenuListener will be the first of the two

                        ml = cml.ml1;
                        prev = cml;
                    }
                }
            }
        }
    }

    /**
     * Adds a non-selectable item to the menu.  The text may be an
     * {@link AttributedText} object or a plain string.
     *
     * @param text the menu text
     * @throws IllegalArgumentException if the text is not a valid type.
     * @see #add(Object, Image, String, MenuAction, Menu)
     */
    public void add(Object text) {
        add(text, null, null, null, null);
    }

    /**
     * Adds a selectable item to the menu.  The text may be an
     * {@link AttributedText} object or a plain string.
     *
     * @param text the menu text
     * @param submenu the submenu
     * @throws IllegalArgumentException if the text is not a valid type.
     * @see #add(Object, Image, String, MenuAction, Menu)
     */
    public void add(Object text, Menu submenu) {
        add(text, null, null, null, submenu);
    }

    /**
     * Adds a non-selectable item to the menu.
     *
     * @param text the menu text
     * @param action this is executed when the specified menu is selected
     * @throws IllegalArgumentException if the text is not a valid type.
     * @see #add(Object, Image, String, MenuAction, Menu)
     */
    public void add(Object text, MenuAction action) {
        add(text, null, null, action, null);
    }

    /**
     * Adds a new item to the menu.  If the submenu is not <code>null</code>
     * then the item will be selectable.  The text may be an
     * {@link AttributedText} object or a plain string.
     * <p>
     * If the submenu already has a parent, then it will be replaced with this
     * menu.</p>
     * <p>
     * When the menu is selected via <em>forward</em>, the specified action is
     * executed.  It may be <code>null</code> if no action is to be executed.
     * </p>
     *
     * @param text the menu text
     * @param icon the menu icon, may be <code>null</code>
     * @param audio the item's audio
     * @param action this is executed when the specified menu is selected, may
     *               be <code>null</code>
     * @param submenu the submenu, may be <code>null</code>
     * @throws NullPointerException if the text is <code>null</code>.
     * @throws IllegalArgumentException if the text is not a valid type.
     * @see #getParent()
     */
    public void add(Object text, Image icon, String audio,
                    MenuAction action, Menu submenu) {
        if (text == null) throw new NullPointerException();
        checkText(text);

        int index = items.size();
        items.addElement(new MenuItem(index, text, icon, audio, submenu != null, action, submenu, null));

        // Mark the menu as changed

        browseList = null;

        // Set the submenu's parent

        if (submenu != null) submenu.parent = this;
    }

    // Other List Management Methods

    /**
     * Removes the menu item at the specified index.
     *
     * @param index index at which to remove the item
     * @throws IndexOutOfBoundsException if the index is out of range.
     * @see #getItemCount()
     * @since 2.5
     */
    public void remove(int index) {
        items.removeElementAt(index);
    }

    // Non-add, remove, or insert methods

    /**
     * Gets the item count.
     *
     * @return the item count.
     */
    public int getItemCount() {
        return items.size();
    }

    /**
     * Sets the specified menu as {@linkplain Bookmarkable bookmarkable}.  If
     * <code>b</code> is <code>null</code> then the item will no longer be
     * bookmarkable.  If the item is already bookmarkable then the information
     * will be replaced by the given {@link Bookmarkable} parameter.
     *
     * @param index the item index
     * @param b the bookmarking information, <code>null</code> to unset
     * @throws IndexOutOfBoundsException if the index is out of range.
     * @see Bookmarkable
     * @see BookmarkInfo
     * @see #getItemCount()
     */
    public void setBookmarkable(int index, Bookmarkable b) {
        BrowseList.Item item = (BrowseList.Item)items.elementAt(index);

        boolean changed = false;

        if (b == null) {
            // Make as not bookmarkable

            if (item instanceof BookmarkableMenuItem) {
                item = new MenuItem(index, (BookmarkableMenuItem)item);
                items.setElementAt(item, index);

                changed = true;
            }
            // If it's not already bookmarkable then do nothing
        } else {
            // Setting the bookmarkable

            if (item instanceof BookmarkableMenuItem) {
                // Replace

                ((BookmarkableMenuItem)item).b = b;
                changed = true;
            } else {
                // Make as bookmarkable

                item = new BookmarkableMenuItem((MenuItem)item, b);
                items.setElementAt(item, index);
                changed = true;
            }
        }

        if (changed) {
            // Mark the menu as changed

            browseList = null;
        }
    }

    /**
     * A convenience method that sets the last menu item's bookmark info.
     * This does nothing if the menu is empty.
     *
     * @param b the new bookmark information
     * @see #setBookmarkable(int, Bookmarkable)
     * @see #getItemCount()
     */
    public void setLastItemBookmarkable(Bookmarkable b) {
        int size = items.size();
        if (size <= 0) return;

        setBookmarkable(size - 1, b);
    }

    /**
     * Sets the audio to be played when the specified menu item has the focus.
     *
     * @param index set the audio for the item at this index
     * @param audio the new audio, may be <code>null</code>
     * @throws IndexOutOfBoundsException if the index is out of range.
     * @see #getItemCount()
     */
    public void setAudio(int index, String audio) {
        MenuItem item = (MenuItem)items.elementAt(index);
        item.audio = audio;
    }

    /**
     * Sets the text for the specified menu item.  The text may be an
     * {@link AttributedText} object or a plain string.
     *
     * @param index set the text of the item at this index
     * @param text the new menu text
     * @throws IllegalArgumentException if the text is not a valid type.
     * @throws IndexOutOfBoundsException if the index is out of range.
     * @see #getItemCount()
     */
    public void setText(int index, Object text) {
        if (text == null) throw new NullPointerException();
        checkText(text);

        MenuItem item = (MenuItem)items.elementAt(index);
        item.text = text;
    }

    /**
     * Sets the item at the specified index as selectable or not selectable,
     * depending on the flag.  If the flag is <code>true</code> the item will
     * be set as selectable regardless if there is a submenu at that location.
     * On a {@link BrowseList}, selectable items will be painted with a right
     * arrow.
     *
     * @param index set the selectable state of the item at this index
     * @param flag whether the item is selectable
     * @throws IndexOutOfBoundsException if the index is out of range.
     * @see #getItemCount()
     */
    public void setSelectable(int index, boolean flag) {
        MenuItem item = (MenuItem)items.elementAt(index);
        item.selectable = flag;
    }

    /**
     * Returns whether the item at the specified index is selectable.  An item
     * will be selectable if it has a submenu or if it was set as
     * non-selectable via {@link #setSelectable(int, boolean) setSelectable}.
     *
     * @param index the item index
     * @return whether the item at the specified index is selectable.
     * @throws IndexOutOfBoundsException if the index is out of range.
     * @see #getItemCount()
     */
    public boolean isSelectable(int index) {
        return ((MenuItem)items.elementAt(index)).selectable;
    }

    /**
     * Sets the icon for the item at the specified index.
     *
     * @param index set the icon for the item at this index
     * @param icon the new icon, may be <code>null</code>
     * @throws IndexOutOfBoundsException if the index is out of range.
     * @see #getItemCount()
     */
    public void setIcon(int index, Image icon) {
        MenuItem item = (MenuItem)items.elementAt(index);
        item.icon = icon;
    }

    /**
     * Sets the default sound to play when the user navigates to a parent
     * menu.  This usually happens when the left NavPlus arrow is tapped.
     * The sound is played before the parent's {@link MenuListener#itemChanged(Menu, int, int, boolean) itemChanged}
     * handler is called.  However, the sound is not played if there is no
     * parent menu.  This is set to <code>null</code> by default.
     *
     * @param clip the new default <em>back</em> sound clip, <code>null</code>
     *             to unset
     */
    public static void setDefaultBackAPM(String clip) {
        defaultBackAPM = clip;
    }

    /**
     * Sets the default sound to play when the user navigates to a submenu.
     * This usually happens when the right NavPlus arrow is tapped.
     * The sound is played after the current menu's
     * {@link MenuListener#itemSelected(Menu, int) itemSelected} and
     * {@link MenuAction#execute(Menu, int) execute} handlers are called.
     * However, the sound is not played if there is no submenu.  This is set
     * to <code>null</code> by default.
     *
     * @param clip the new default <em>forward</em> sound clip,
     *             <code>null</code> to unset
     */
    public static void setDefaultForwardAPM(String clip) {
        defaultForwardAPM = clip;
    }

    /**
     * Sets the default sound to play when the user selects a menu item having
     * no submenu.  This usually happens when the right NavPlus arrow is
     * tapped.  If the current item does not have a submenu then the sound is
     * played before current menu's
     * {@link MenuListener#itemSelected(Menu, int) itemSelected} and
     * {@link MenuAction#execute(Menu, int) execute} handlers are called.
     * This is set to <code>null</code> by default.
     *
     * @param clip the new default <em>select</em> sound clip,
     *             <code>null</code> to unset
     */
    public static void setDefaultSelectAPM(String clip) {
        defaultSelectAPM = clip;
    }

    /**
     * Sets the sound to play when the user navigates to a parent menu.  This
     * overrides the {@link #setDefaultBackAPM(String) default sound}.  If
     * this is set to <code>null</code> then the default sound will be used.
     * This is set to <code>null</code> by default.
     *
     * @param clip the new <em>back</em> sound clip, <code>null</code> to unset
     * @see #setDefaultBackAPM(String)
     */
    public void setBackAPM(String clip) {
        backAPM = clip;
    }

    /**
     * Sets the sound to play when the user navigates to a submenu.  This
     * overrides the {@link #setDefaultForwardAPM(String) default sound}.  If
     * this is set to <code>null</code> then the default sound will be used.
     * This is set to <code>null</code> by default.
     *
     * @param clip the new <em>forward</em> sound clip, <code>null</code> to
     *             unset
     * @see #setDefaultForwardAPM(String)
     */
    public void setForwardAPM(String clip) {
        forwardAPM = clip;
    }

    /**
     * Sets the sound to play when the user selects a menu item having no
     * submenu.  This overrides the {@link #setDefaultSelectAPM(String) default sound}.
     * If this is set to <code>null</code> then the default sound will be
     * used. This is set to <code>null</code> by default.
     *
     * @param clip the new <em>select</em> sound clip, <code>null</code> to
     *             unset
     * @see #setDefaultSelectAPM(String)
     */
    public void setSelectAPM(String clip) {
        selectAPM = clip;
    }

    /**
     * Sets whether to play the <em>back</em> sound.  This is useful for when
     * the default sound is set but only one or a few menus don't want to have
     * the sound play.  If this is set to <code>false</code> then the sound
     * will never play.  The default is <code>true</code>.
     *
     * @param flag whether to play any <em>back</em> sound
     */
    public void setPlayBackAPM(boolean flag) {
        playBackAPM = flag;
    }

    /**
     * Sets whether to play the <em>back</em> sound.  This is useful for when
     * the default sound is set but only one or a few menus don't want to have
     * the sound play.  If this is set to <code>false</code> then the sound
     * will never play.  The default is <code>true</code>.
     *
     * @param flag whether to play any <em>back</em> sound
     */
    public void setPlayForwardAPM(boolean flag) {
        playForwardAPM = flag;
    }

    /**
     * Sets whether to play the <em>back</em> sound.  This is useful for when
     * the default sound is set but only one or a few menus don't want to have
     * the sound play.  If this is set to <code>false</code> then the sound
     * will never play.  The default is <code>true</code>.
     *
     * @param flag whether to play any <em>select</em> sound
     */
    public void setPlaySelectAPM(boolean flag) {
        playSelectAPM = flag;
    }

    /**
     * Adds an action for the item at the specified index.
     *
     * @param index set the action for the item at this index
     * @param action the new action, may be <code>null</code>
     * @throws IndexOutOfBoundsException if the index is out of range.
     * @see #getItemCount()
     */
    public void addAction(int index, final MenuAction action) {
        final MenuItem item = (MenuItem)items.elementAt(index);

        // Chain the actions

        if (item.action == null) {
            item.action = action;
        } else if (action != null) {
            final MenuAction a = item.action;

            item.action = new MenuAction() {
                    public void execute(Menu menu, int index) {
                        a.execute(menu, index);
                        action.execute(menu, index);
                    }
                };
        }
    }

    /*public void add(Displayable d, MenuAction action, Menu submenu) {
        labels.addElement(d);
        submenus.addElement(submenu);
        actions.addElement(action);

        // Mark the menu as changed

        browseList = null;

        // Set the submenu's parent

        if (submenu != null) submenu.parent = this;
    }*/

    /**
     * Sets the parent for this menu to be the specified menu but does not add
     * this menu to the parent.  The new parent can be <code>null</code>.
     * <p>
     * This method is useful when it is necessary to back out of a menu, say
     * from a transient result, and not be able to navigate back.</p>
     *
     * @param parent the new parent for this menu
     */
    public void setOneWayParent(Menu parent) {
        this.parent = parent;
    }

    /**
     * Attaches some arbitrary metadata to this menu.
     * <p>
     * This can be useful, for example, when a listener object common to
     * several menus needs to quickly determine the identity of the menu it
     * was handed.</p>
     *
     * @param data the new metadata, may be <code>null</code>
     * @see #getMetadata()
     */
    public void setMetadata(Object data) {
        this.metadata = data;
    }

    /**
     * Returns the metadata associated with this menu.
     *
     * @return the metadata associated with this menu, may be
     *         <code>null</code>.
     * @see #setMetadata(Object)
     */
    public Object getMetadata() {
        return metadata;
    }

    /**
     * Attaches some arbitrary metadata to a specific menu item.
     *
     * @param index the item index
     * @param data the new metadata, may be <code>null</code>
     * @throws IndexOutOfBoundsException if the index is out of range.
     * @see #getItemCount()
     */
    public void setMetadata(int index, Object data) {
        MenuItem item = (MenuItem)items.elementAt(index);
        item.metadata = data;
    }

    /**
     * Gets the metadata associated with the specified menu item.
     *
     * @param index the item index
     * @return the metadata associated with the menu item, may be
     *         <code>null</code>.
     * @throws IndexOutOfBoundsException if the index is out of range.
     * @see #getItemCount()
     */
    public Object getMetadata(int index) {
        MenuItem item = (MenuItem)items.elementAt(index);
        return item.metadata;
    }

    /**
     * Returns the parent menu, or <code>null</code> if this menu has no
     * parent.  A menu can have only one parent.
     *
     * @return the parent menu, or <code>null</code> if this menu has no
     *         parent.
     * @see #add(Object, Image, String, MenuAction, Menu)
     */
    public Menu getParent() {
        return parent;
    }

    /**
     * Gets the submenu for the item at the specified index.  If the item does
     * not have a submenu then this returns <code>null</code>.
     *
     * @param index get the submenu for the item at this index
     * @return the submenu for the specified menu item, or <code>null</code>
     *         if the item does not have a submenu.
     * @throws IndexOutOfBoundsException if the index is out of range.
     * @see #getItemCount()
     */
    public Menu getSubmenu(int index) {
        MenuItem item = (MenuItem)items.elementAt(index);
        return item.submenu;
    }

    /**
     * Sets the submenu at the specified index.  A submenu can be removed by
     * setting it to <code>null</code>.
     *
     * @param index sets the submenu at this index
     * @param submenu the new submenu, may be <code>null</code> to remove
     * @throws IndexOutOfBoundsException if the index is out of range.
     * @see #getItemCount()
     * @since 2.5
     */
    public void setSubmenu(int index, Menu submenu) {
        MenuItem item = (MenuItem)items.elementAt(index);
        item.submenu = submenu;
    }

    /**
     * Converts a string or {@link AttributedText} to an unattributed string.
     *
     * @param text convert this text
     * @return the text as a string.
     */
    private static String toString(Object text) {
        if (text instanceof String) return (String)text;

        // Convert AttributedText

        AttributedText at = (AttributedText)text;

        StringBuffer buf = new StringBuffer();
        for (int i = 0, count = at.getCount(); i < count; i++) {
            buf.append(at.getText(i));
        }

        return buf.toString();
    }

    /**
     * This gives subclasses the opportunity to use their own
     * {@link BrowseList} implementation, for example to paint icons
     * differently.
     * <p>
     * This is used by {@link #getBrowseList()}.</p>
     *
     * @param items the items
     * @param title the title
     * @return a new {@link MenuBrowseList}.
     */
    protected MenuBrowseList createBrowseList(Vector items, BrowseList.Title title) {
        return new MenuBrowseList(items, title);
    }

    /**
     * Gets the browse list display object.  If the menu has not changed since
     * the last call to this method then the previous browse list will be
     * returned.
     * <p>
     * This uses {@link #createBrowseList(Vector, BrowseList.Title)} to create
     * the browse list.</p>
     * <p>
     * The browse list returned by this method should not be used to show the
     * menu, as some internal state may not get updated properly and listeners
     * may not get properly notified.  Instead, use the
     * {@link #show(Display, Transition)} method.</p>
     *
     * @return the browse list display object.
     * @see #createBrowseList(Vector, BrowseList.Title)
     * @see #show(Display, Transition)
     */
    protected BrowseList getBrowseList() {
        if (browseList == null) {
            BrowseList.Title title = (this.title == null)
                                     ? null
                                     : new BrowseList.Title(toString(this.title), null);

            // Create the BrowseList

            browseList = createBrowseList(items, title);
        }

        // NOTE Don't set the focus item here
        return browseList;
    }

    /**
     * Notifies the listener that an item was changed.
     *
     * @param oldIndex the old index
     * @param newIndex the new index
     * @param activated whether the menu was activated
     * @see MenuListener#itemChanged(Menu, int, int, boolean)
     */
    protected void fireItemChanged(int oldIndex, int newIndex, boolean activated) {
        if (listener != null) {
            listener.itemChanged(this, oldIndex, newIndex, activated);
        }
    }

    /**
     * Notifies the listener that an item was selected.
     *
     * @param index the item index
     * @see MenuListener#itemSelected(Menu, int)
     */
    protected void fireItemSelected(int index) {
        if (listener != null) {
            listener.itemSelected(this, index);
        }
    }

    /**
     * Notifies the listener that the menu was shown on the display.
     *
     * @see MenuListener#menuShown(Menu)
     */
    protected void fireMenuShown() {
        if (listener != null) {
            listener.menuShown(this);
        }
    }

    /**
     * Sets the item index that should have the focus.  Note that currently,
     * browse lists do not support setting the current item to the title, so
     * the index must be &ge; zero.
     * <p>
     * The menu listener will be notified with an <em>item-changed</em> event
     * and with its <em>activated</em> parameter set to <code>false</code>.</p>
     *
     * @param index the new index
     * @throws IndexOutOfBoundsException if the index is out of range.
     * @see #getItemCount()
     * @see MenuListener#itemChanged(Menu, int, int, boolean)
     */
    public void setFocusIndex(int index) {
        if (index < 0 || getItemCount() <= index) {
            throw new IndexOutOfBoundsException("Index out of range: " + index);
        }

        int oldIndex = this.focusIndex;
        this.focusIndex = index;
        fireItemChanged(oldIndex, index, false);
    }

    /**
     * Gets the index of the item that currently has the focus.  This will
     * return -1 if the title is showing.
     *
     * @return the index of the currently focused item.
     */
    public int getFocusIndex() {
        return focusIndex;
    }

    /**
     * Shows the menu on the specified display using the specified transition.
     * The display's current transition will not be changed after this method
     * completes.
     * <p>
     * The menu listener will be notified with an <em>item-changed</em> event
     * and with its <em>activated</em> parameter set to <code>true</code>.
     * The notification will happen just before it is made active on the
     * display.</p>
     * <p>
     * After the menu is shown on the display, a <em>menu-shown</em> event
     * is sent.</p>
     *
     * @param display the display
     * @param t the transition
     * @see Transition
     * @see MenuListener#itemChanged(Menu, int, int, boolean)
     * @see MenuListener#menuShown(Menu)
     */
    public void show(Display display, Transition t) {
        Transition oldT = display.getTransition();

        BrowseList bl = getBrowseList();
        int oldIndex = bl.getFocusIndex();

        if (oldIndex != focusIndex) {
            dontNotifyItemChanged = true;
            bl.setFocusItem(focusIndex);
            dontNotifyItemChanged = false;
        }

        // Notify the listener

        fireItemChanged(oldIndex, focusIndex, true);

        display.setTransition(t);
        display.setCurrent(bl);

        // Restore the current transition

        display.setTransition(oldT);

        fireMenuShown();
    }

    /**
     * Selects an item as if it was chosen with a tap on the NavPlus right
     * arrow.  This returns the new menu if there is a submenu and
     * <code>this</code> otherwise.
     *
     * @param index select this menu item
     * @param display use this display
     * @param player an optional media player
     * @return the submenu if there is one, and <code>this</code> if there is
     *         no submenu.
     * @throws IndexOutOfBoundsException if the index is out of range.
     * @see #getItemCount()
     * @see #setForwardAPM(String)
     */
    public Menu doSelect(int index, Display display, MediaPlayer player) {
        MenuItem item = (MenuItem)items.elementAt(index);

        // Play a sound

        if (item.submenu == null) {
            if (playSelectAPM) {
                play(player, selectAPM != null ? selectAPM : defaultSelectAPM);
            }
        }

        // Execute any actions

        MenuAction action = item.action;
        if (action != null) {
            action.execute(this, index);
        }
        fireItemSelected(index);

        Menu sm = item.submenu;
        if (sm != null) {
            if (playForwardAPM) {
                play(player, forwardAPM != null ? forwardAPM : defaultForwardAPM);
            }

            sm.show(display, Transition.RIGHT_TO_LEFT);

            return sm;
        }

        return this;
    }

    /**
     * Handles a menu event by properly displaying the information.  The
     * <code>display</code> and <code>player</code> arguments are the visual
     * and audio UI objects used to indicate any menu effects and changes.
     * <p>
     * This will return itself if we have not changed to a parent menu or
     * submenu.</p>
     *
     * @param event the menu event
     * @param display the display object for setting the browse lists
     * @param player an optional media player
     * @return the new menu, if we have navigated forward or backwards.
     */
    public Menu handleMenuEvent(MenuEvent event, Display display,
                                MediaPlayer player) {
        // Navigate the menu

        BrowseList bl = getBrowseList();

        Menu retval = this;

        switch (event.getId()) {
            case MenuEvent.MENU_UP: {
                int oldIndex = bl.getFocusIndex();
                int newIndex = bl.focusToPrevious();
                focusIndex = newIndex;
                fireItemChanged(oldIndex, newIndex, false);

                break;
            }

            case MenuEvent.MENU_DOWN: {
                int oldIndex = bl.getFocusIndex();
                int newIndex = bl.focusToNext();
                focusIndex = newIndex;
                fireItemChanged(oldIndex, newIndex, false);

                break;
            }

            case MenuEvent.MENU_LEFT:
                Menu parent = getParent();
                if (parent != null) {
                    // DONETODO Potentially play a sound
                    if (playBackAPM) {
                        play(player, backAPM != null ? backAPM : defaultBackAPM);
                    }

                    parent.show(display, Transition.LEFT_TO_RIGHT);

                    retval = parent;
                //} else {
                //    play(player, MENU_TOP_SOUND);
                }
                break;

            case MenuEvent.MENU_RIGHT:
                // Execute the action

                retval = doSelect(focusIndex, display, player);

                break;
        }

        return retval;
    }

    /**
     * Convenience method that checks for a <code>null</code> player and then
     * plays the specified sound.
     *
     * @param player the player, possibly <code>null</code>
     * @param sound play this sound
     */
    private static void play(MediaPlayer player, String sound) {
        if (player != null && sound != null) {
            player.play(sound);
        }
    }

    /**
     * Returns a string representation of this menu.
     *
     * @return a string representation of this menu.
     */
    //@Override
    public String toString() {
        StringBuffer buf = new StringBuffer();

        buf.append("Menu");
        if (title != null) {
            buf.append('(').append(title).append(')');
        }
        buf.append('[');

        // Print the children

        int size = items.size();
        for (int i = 0; i < size; i++) {
            if (i > 0) buf.append(',');

            MenuItem item = (MenuItem)items.elementAt(i);
            buf.append(item.text);

            Menu menu = item.submenu;
            if (menu != null) {
                buf.append('=').append(menu);
            }
        }

        buf.append(']');

        return buf.toString();
    }
}
