/*
 * Date: Apr 23, 2010
 * Time: 1:22:32 PM
 * (c) 2010 Livescribe, Inc.
 */
package com.livescribe.ext.plugins;

import com.livescribe.display.AttributedText;
import com.livescribe.display.Display;
import com.livescribe.display.Transition;
import com.livescribe.display.Font;
import com.livescribe.ext.util.Log;
import com.livescribe.penlet.*;
import com.livescribe.icr.ICRContext;
import com.livescribe.icr.Resource;
import com.livescribe.event.*;
import com.livescribe.afp.PageInstance;
import com.livescribe.geom.Rectangle;
import com.livescribe.geom.Shape;
import com.livescribe.storage.StrokeStorage;
import com.livescribe.ui.ScrollLabel;

import java.util.Vector;

/**
 * A penlet plugin that performs common functions.
 * <ul>
 * <li>Manage the ICR state and results.</li>
 * <li>Display filtered results on the screen.</li>
 * <li>Double-tap detection.</li>
 * </ul>
 * <p>
 * This is designed to work well with applications that use a single line of
 * handwriting and some terminating event (including double-tap) to perform a
 * function.</p>
 * <p>
 * Note that this plugin does not save and restore the screen.</p>
 *
 * @author Shawn Silverman
 */
public class AcquireHWPlugin extends AbstractPenletPlugin {
    private static final String CLASS_NAME = "AcquireHWPlugin";

    /** The default HWR inactivity time. */
    public static final int DEFAULT_HWR_INACTIVITY_TIME = 2000;

    // State

    private static final int START_STATE             = 0;
    private static final int WAIT_FOR_PEN_DOWN_STATE = 1;
    private static final int RUNNING_STATE           = 2;

    /**
     * Done with the actions, but waiting to be deactivated.  Events should be
     * consumed in this state.
     */
    private static final int FINISHED_STATE    = 3;

    // Errors

    /**
     * The plugin has not been used to generate a result or it has been
     * deactivated early.
     */
    public static final int RESULT_UNDEFINED    = 0;

    /** The result is "OK". */
    public static final int RESULT_OK           = 1;

    /**
     * An overlap was detected and overlapping was set to <em>not allowed</em>.
     *
     * @see #setAllowOverlap(boolean)
     */
    public static final int RESULT_OVERLAPPED   = 2;

    /**
     * The page was changed in the middle of the recognition and page changing
     * was set to <em>not allowed</em>.
     *
     * @see #setAllowPageChange(boolean)
     */
    public static final int RESULT_PAGE_CHANGED = 3;

    /** Indicates that an HWR error occurred. */
    public static final int RESULT_HWR_ERROR = 4;

    // Messages

    //private static final String PAGE_CHANGED_MSG = "Page changed. Try again.";
    //private static final String ENTERED_OLD_REGION_MSG = "Draw in blank area. Try again.";

    // UI

    private ScrollLabel label;

    // Plugin State

    private int state;

    private final Object lock = new Object();

    // Stroke State

    /**
     * This is set by {@link MyHWRListener#hwrUserPause(long, String)} and
     * {@link MyHWRListener#hwrResult(long, String)} to be the latest stable
     * result.
     */
    private String hwrResult;

    /**
     * Contains any error string given by {@link HWRListener#hwrError(long, String)}.
     * This will only be valid if the result status is set to
     * {@link #RESULT_HWR_ERROR}.
     */
    private String hwrErrorString;

    /** The bounds of the HWR result. */
    private Rectangle hwrResultBounds;

    /**
     * Indicates that the next stroke should be recorded.  This is managed by
     * the <em>pen-down</em> event.
     */
    private boolean recordNextStroke;

    // Shape State

    /** The current bounding box of any received strokes. */
    protected Rectangle currBB;

    /** The current page, set when a <em>pen-down</em> event occurs. */
    protected PageInstance currPage;

    // HWR

    /**
     * The HWR inactivity time.  The default is
     * {@value #DEFAULT_HWR_INACTIVITY_TIME}&nbsp;ms.
     */
    private int hwrInactivityTime = DEFAULT_HWR_INACTIVITY_TIME;

    /** The HWR component. */
    private ICRContext icrContext;

    /** The HWR listener. */
    private HWRListener myHWRListener = new MyHWRListener();

    // The ICR Resources

    private Vector akRes;
    private Vector skRes;
    private Vector lkRes;
    private Vector appRes;

    // Other state

    private boolean displayOnResult;
    private DisplayFilter displayFilter;

    private boolean allowOverlap;
    private boolean allowPageChange;
    private boolean doubleTapTerminates;
    private boolean userPauseTerminates;

    private boolean seenDoubleTap;

    private boolean strokePending;

    /** The state of the current result. */
    private int resultStatus;

    /**
     * An interface that defines how to display the results on the screen when
     * {@link #setDisplayOnResult(boolean, DisplayFilter) displaying the results}
     * is enabled.
     * <p>
     * If no display filter is set, then the result is simply used as-is.</p>
     *
     * @see #setDisplayOnResult(boolean, DisplayFilter)
     */
    public interface DisplayFilter {
        /**
         * This filters the result for use when HWR results are displayed on
         * the screen.
         * <p>
         * This should return an instance of {@link AttributedText} or
         * {@link String}.  If the object returned is not an instance of
         * either of those classes then the returned object's
         * {@link Object#toString()} is use instead.</p>
         *
         * @param result the HWR result
         * @return the filtered text for display onscreen.
         * @see AttributedText#parseEnriched(String, com.livescribe.display.Font)
         * @see #setDisplayOnResult(boolean, DisplayFilter)
         */
        public Object filter(String result);
    }

    /**
     * Implementation of {@link HWRListener} in order to avoid public exposure
     * of the methods.
     */
    private final class MyHWRListener implements HWRListener {
        /**
         * This is called when some handwriting is recognized.
         * <p>
         * The following tasks are performed.</p>
         * <ol>
         * <li>{@link AcquireHWPlugin#hwrResult} is set to the latest result, and</li>
         * <li>The result is drawn to the screen if it is not empty.</li>
         * </ol>
         *
         * @param time the event time
         * @param result the handwriting that has been recognized since all new
         *               strokes were sent to the HWR engine
         */
        public void hwrResult(long time, String result) {
            Log.debug(CLASS_NAME, "hwrResult", "result[" + result.length() + "]=" + result);

            synchronized (lock) {
                hwrResult = result;
                resultStatus = RESULT_OK;

                // TODO Wait for the RUNNING state when showing the result because a restart may have reset the state to WAIT_FOR_PEN_DOWN and a subclass may have a message on the screen
                if (displayOnResult) {
                    drawResult(result);
                }

                if (doubleTapTerminates && seenDoubleTap) {
                    changeState(FINISHED_STATE);

                    // Deactivate if we have no stroke pending
                    // Otherwise, let the deactivation happen when we receive the
                    // stroke

                    if (!strokePending) {
                        deactivate();
                    }
                }
            }
        }

        /**
         * The user has paused while entering strokes into the HWR engine.
         * <p>
         * The following tasks are performed:</p>
         * <ol>
         * <li>{@link AcquireHWPlugin#hwrResult} is set to the result,</li>
         * <li>The display is updated if required,</li>
         * <li>If we decide to terminate here, finish and, if there are no pending
         *     strokes, deactivate.
         * </ol>
         *
         * @param time the event time
         * @param result the current HWR result
         */
        public void hwrUserPause(long time, String result) {
            Log.debug(CLASS_NAME, "hwrUserPause", "result[" + result.length() + "]=" + result);

            synchronized (lock) {
                hwrResult = result;
                hwrResultBounds = icrContext.getTextBoundingBox();
                resultStatus = RESULT_OK;

                // TODO Wait for the RUNNING state when showing the result because a restart may have reset the state to WAIT_FOR_PEN_DOWN and a subclass may have a message on the screen
                if (displayOnResult) {
                    drawResult(result);
                }

                // Terminate if double taps don't terminate or user pauses do

                if (!doubleTapTerminates || seenDoubleTap || userPauseTerminates) {
                    changeState(FINISHED_STATE);

                    // Deactivate if we have no stroke pending
                    // Otherwise, let the deactivation happen when we receive the
                    // stroke

                    if (!strokePending) {
                        deactivate();
                    }
                }
            }
        }

        /**
         * Called when an error occurs during HWR.
         * <p>
         * This implementation does nothing.</p>
         *
         * @param time the event time
         * @param error the error message
         */
        public void hwrError(long time, String error) {
            Log.error(CLASS_NAME + "$MyHWRListener", "hwrError", "[" + time + "]: " + error);

            synchronized (lock) {
                resultStatus = RESULT_HWR_ERROR;
                hwrErrorString = error;

                changeState(FINISHED_STATE);

                // Deactivate if we have no stroke pending
                // Otherwise, let the deactivation happen when we receive the
                // stroke

                if (!strokePending) {
                    deactivate();
                }
            }
        }

        /**
         * Called when the user crosses out text.
         * <p>
         * This implementation does nothing.</p>
         *
         * @param time the event time
         * @param result the current HWR result
         */
        public void hwrCrossingOut(long time, String result) {
        }
    }

    /**
     * Creates a new HWR plugin.  The plugin is set to not consume all events.
     * In other words, the superclass constructor is called with
     * <code>consumeAllEvents</code> set to <code>false</code>.
     *
     * @param penlet the associated penlet
     * @see AbstractPenletPlugin#AbstractPenletPlugin(Penlet, boolean)
     */
    public AcquireHWPlugin(Penlet penlet) {
        super(penlet, false);  // Don't consume all events
    }

    // Penlet Lifecycle Methods

    /**
     * Activates the plugin.  This creates a new ICR context.  Note that this
     * does not save the display.
     * <p>
     * If the HWR engine could not be initialized or if the ICR resources have
     * not been set then this will throw a {@link PenletStateChangeException}.
     * </p>
     * <p>
     * The plugin will start processing events when it sees the next
     * <em>pen-down</em> event.  It will ignore all other pen tip and stroke
     * events&mdash;and pass them along to the application&mdash;until then.
     * </p>
     *
     * @see #setHWRInactivityTime(int)
     * @see #addAKSystemResource(String)
     * @see #addSKSystemResource(String)
     * @see #addLKSystemResource(String)
     * @see #addAppResource(String)
     * @throws PenletStateChangeException if there was an error initializing
     *         HWR engine or if the ICR resources were not set.
     */
    public void activate() throws PenletStateChangeException {
        if (active) return;

        changeState(START_STATE);

        if (akRes == null && skRes == null && lkRes == null && appRes == null) {
            throw new PenletStateChangeException("ICR resources not set");
        }

        // Configure the ICR context

        try {
            icrContext = getPenlet().getContext().getICRContext(hwrInactivityTime, myHWRListener);
            icrContext.addResourceSet(createResources(icrContext));
        } catch (RuntimeException ex) {
            // Dispose of the ICR context right away

            if (icrContext != null) {
                icrContext.dispose();
                icrContext = null;
            }

            throw new PenletStateChangeException(ex);
        }

        // UI

        label = new ScrollLabel();

        restart(true);

        //saveDisplay();
        super.activate();
    }

    /**
     * Restarts the handwriting recognition process.  This can be used from
     * inside a subclass's {@link #deactivate()} implementation if actual
     * deactivation is not yet required.
     * <p>
     * There are two ways to restart the process.  The first way is to
     * completely reset the state so that new handwriting will not continue
     * from where the recognition left off.  If this is the desired behavior,
     * pass <code>true</code> for the <code>resetState</code> parameter.
     * <p>
     * The other way is to continue the recognition from where it left off.
     * If this is desired then pass <code>false</code> for the parameter.</p>
     *
     * @param resetState indicates whether to reset the state
     */
    protected void restart(boolean resetState) {
        synchronized (lock) {
            // Other

            recordNextStroke = true;

            // State

            resultStatus = RESULT_UNDEFINED;
            if (resetState) resetState();
            changeState(WAIT_FOR_PEN_DOWN_STATE);
        }
    }

    /**
     * Deactivates the plugin.  This disposes of the ICR context.  Note that
     * this does not restore the display.
     * <p>
     * This can be overridden and used as a way to detect when this plugin is
     * finished recognition.  The HWR process can be restarted via
     * {@link #restart(boolean)}.  In this case, don't call
     * <code>super.deactivate()</code> because that would actually deactivate
     * the plugin.</p>
     */
    public void deactivate() {
        if (!active) return;

        //restoreDisplay();

        // ICR

        if (icrContext != null) {
            icrContext.dispose();
            icrContext = null;
        }

        super.deactivate();
    }

    // State

    /**
     * Changes the state.
     *
     * @param state the new state
     */
    private void changeState(int state) {
        this.state = state;
    }

    // Properties

    /**
     * Sets the HWR timeout, in milliseconds.  The timeout is measured from
     * when the user lifts the pen.
     * <p>
     * The default is {@value #DEFAULT_HWR_INACTIVITY_TIME}&nbsp;ms.</p>
     *
     * @param time the new HWR timeout, in ms
     * @see #DEFAULT_HWR_INACTIVITY_TIME
     */
    public void setHWRInactivityTime(int time) {
        this.hwrInactivityTime = time;
    }

    /**
     * Creates an array of {@link Resource}s using all the added resources.
     *
     * @param icr the ICR context to use
     * @return a new array of {@link Resource}s.
     */
    private Resource[] createResources(ICRContext icr) {
        Vector v = new Vector(3);

        if (akRes != null) {
            for (int i = 0; i < akRes.size(); i++) {
                v.addElement(icr.createAKSystemResource((String)akRes.elementAt(i)));
            }
        }
        if (skRes != null) {
            for (int i = 0; i < skRes.size(); i++) {
                v.addElement(icr.createSKSystemResource((String)skRes.elementAt(i)));
            }
        }
        if (lkRes != null) {
            for (int i = 0; i < lkRes.size(); i++) {
                v.addElement(icr.createLKSystemResource((String)lkRes.elementAt(i)));
            }
        }
        if (appRes != null) {
            for (int i = 0; i < appRes.size(); i++) {
                v.addElement(icr.createAppResource((String)appRes.elementAt(i)));
            }
        }

        Resource[] res = new Resource[v.size()];
        v.copyInto(res);
        return res;
    }

    /**
     * Adds an Alphabet Knowledge system ICR resource.
     *
     * @param res add this AK system resource
     * @see ICRContext#createAKSystemResource(String)
     * @see #clearResources()
     */
    public void addAKSystemResource(String res) {
        if (akRes == null) {
            akRes = new Vector(2);
        }
        akRes.addElement(res);
    }

    /**
     * Adds a Subset Knowledge system ICR resource.
     *
     * @param res add this SK system resource
     * @see ICRContext#createSKSystemResource(String)
     * @see #clearResources()
     */
    public void addSKSystemResource(String res) {
        if (skRes == null) {
            skRes = new Vector(2);
        }
        skRes.addElement(res);
    }

    /**
     * Adds a Lexical Knowledge system ICR resource.
     *
     * @param res add this LK system resource
     * @see ICRContext#createLKSystemResource(String)
     * @see #clearResources()
     */
    public void addLKSystemResource(String res) {
        if (lkRes == null) {
            lkRes = new Vector(2);
        }
        lkRes.addElement(res);
    }

    /**
     * Adds an application-specific ICR resource.
     *
     * @param res add this application resource
     * @see ICRContext#createAppResource(String)
     * @see #clearResources()
     */
    public void addAppResource(String res) {
        if (appRes == null) {
            appRes = new Vector(2);
        }
        appRes.addElement(res);
    }

    /**
     * Clears all the ICR resources.
     *
     * @see #addAKSystemResource(String)
     * @see #addSKSystemResource(String)
     * @see #addLKSystemResource(String)
     * @see #addAppResource(String)
     */
    public void clearResources() {
        appRes = null;

        lkRes  = null;
        skRes  = null;
        akRes  = null;
    }

    /*
     * Sets the ICR resources to use for HWR.  Note that the resources cannot
     * changed while the plugin is active.  Any changes made here would take
     * effect when the plugin is next activated.
     * <p>
     * This makes a shallow copy of the given array.</p>
     *
     * @param resources the new ICR resources
     * @throws NullPointerException if the parameter is <code>null</code>.
     * @see ICRContext
     * @see Resource
     * @see #isActive()
     *
    public void setICRResources(String[] resources) {
        this.icrResources = new String[resources.length];

        System.arraycopy(resources, 0, this.icrResources, 0, resources.length);
    }*/

    /*
     * Sets the region information to use when assigning a recognized written
     * area
     * @param r
     *
    public void setRegionInfo(Region r) {
        new Region()
    }*/

    /**
     * Sets whether strokes are allowed on regions that are not open paper.  A
     * region is considered as "not open paper" if it has no assigned meaning,
     * to the current app as a dynamic or fixed print control, for example.
     * <p>
     * The default is to disallow overlap, or <code>false</code>.</p>
     *
     * @param flag the new state
     */
    public void setAllowOverlap(boolean flag) {
        allowOverlap = flag;
    }

    /**
     * Sets whether to allow handwriting recognition over multiple pages.
     * The default is to disallow this, or <code>false</code>.
     *
     * @param flag the new state
     */
    public void setAllowPageChange(boolean flag) {
        allowPageChange = flag;
    }

    /**
     * Sets whether to display the results as the are received.  The default
     * is to not display intermediate results, or <code>false</code>.
     * <p>
     * This also sets the display filter to use when displaying received
     * results.  If this is set to <code>null</code> then the result will be
     * used as-is. The default is no display filter, or <code>null</code>.</p>
     *
     * @param flag the new state
     * @param displayFilter the new display filter, may be <code>null</code>
     * @see DisplayFilter
     */
    public void setDisplayOnResult(boolean flag, DisplayFilter displayFilter) {
        displayOnResult = flag;
        this.displayFilter = displayFilter;
    }

    /**
     * Sets whether the plugin should terminate when a user timeout occurs or
     * if it should terminate after a double tap.  If set to
     * <code>true</code>, the plugin will terminate after a double tap.  If
     * set to <code>false</code>, the plugin will terminate after it receives
     * its first timeout.
     * <p>
     * The default is <code>false</code>.</p>
     * <p>
     * If this is set to <code>true</code> then double taps are not consumed
     * by this plugin and are passed along to the application.  This is to
     * allow the application to, for example, alert the user.  Note that there
     * is no guarantee as to the ordering of this event with respect to plugin
     * deactivation.</p>
     *
     * @param flag the new state
     * @see #setUserPauseTerminates(boolean)
     */
    public void setDoubleTapTerminates(boolean flag) {
        doubleTapTerminates = flag;
    }

    /**
     * Sets whether a user pause can terminate the plugin.  If a double
     * tap is set to <em>not</em> terminate the plugin
     * ({@link #setDoubleTapTerminates(boolean)} set to <code>false</code>),
     * then this setting is ignored and it will be as if it is set to
     * <code>true</code>.  In other words, the only time a user pause does not
     * terminate the plugin is when this setting is set to <code>false</code>
     * and {@link #setDoubleTapTerminates(boolean)} is set to
     * <code>true</code>.
     * <p>
     * The default is <code>false</code>.</p>
     *
     * @param flag the new state
     * @see #setDoubleTapTerminates(boolean)
     */
    public void setUserPauseTerminates(boolean flag) {
        userPauseTerminates = flag;
    }

    /**
     * Returns the last HWR result.  This will return <code>null</code> if the
     * plugin has never been activated before, or if the current result was
     * not recognized.
     *
     * @return the last HWR result.
     * @see #getResultBounds()
     * @see #getResultStatus()
     */
    public String getResult() {
        return hwrResult;
    }

    /**
     * Gets the bounds of the last result.
     *
     * @return the bounds of the last result.
     * @see #getResult()
     */
    public Rectangle getResultBounds() {
        return hwrResultBounds;
    }

    /**
     * Gets the status of the result, for example if the result is
     * <code>null</code> because of an error.
     * <p>
     * Please see the <code>RESULT_<em>XXX</em></code> constants.</p>
     * <p>
     * If this returns {@link #RESULT_HWR_ERROR} then
     * {@link #getHWRErrorString()} can be used to retrieve any associated
     * error string.</p>
     *
     * @return the result status.
     * @see #RESULT_UNDEFINED
     * @see #RESULT_OK
     * @see #RESULT_OVERLAPPED
     * @see #RESULT_PAGE_CHANGED
     * @see #RESULT_HWR_ERROR
     */
    public int getResultStatus() {
        return resultStatus;
    }

    /**
     * Returns the HWR error string if {@link #getResultStatus()} returns
     * {@link #RESULT_HWR_ERROR}.
     *
     * @return the HWR error string.
     * @see HWRListener#hwrError(long, String)
     * @since 2.5
     */
    public String getHWRErrorString() {
        return hwrErrorString;
    }

    // Internal Methods

    /**
     * Draws the result to the screen using any display filter.
     *
     * @param result the HWR result
     */
    private void drawResult(String result) {
        if (displayFilter == null) {
            draw(result, false);
        } else {
            draw(displayFilter.filter(result));
        }
    }

    /**
     * Convenience method that draws text on the screen and sets it as active.
     * The text may be marked up.
     *
     * @param text the text to draw
     * @param parseEnriched indicates whether to attempt parsing the text as
     *                      enriched text
     * @see AttributedText#parseEnriched(String, Font)
     */
    private void draw(String text, boolean parseEnriched) {
        Object o = (parseEnriched)
                ? AttributedText.parseEnriched(text, null)
                : text;
        draw(o);
    }

    /**
     * Convenience method that draws an object to the screen.
     *
     * @param o the text to draw
     */
    private void draw(Object o) {
        if (o instanceof AttributedText) {
            label.draw((AttributedText)o, true);
        } else {
            label.draw(String.valueOf(o), true);
        }

        // Set the label and with the default transition

        Display display = getPenlet().getContext().getDisplay();

        display.setTransition(Transition.DEFAULT);
        if (display.getCurrent() != label) {
            display.setCurrent(label);
        }
    }

    /**
     * Resets any state.  Subclasses should call this if they override this
     * method.
     * <p>
     * The following state is reset:</p>
     * <ol>
     * <li>ICR context is cleared,</li>
     * <li>{@link #currBB} and {@link #currPage} set to <code>null</code>,</li>
     * <li>{@link #hwrResult} and {@link #hwrResultBounds} are set to
     *     <code>null</code>, and</li>
     * <li>{@link #seenDoubleTap} is set to <code>false</code>.</li>
     * </ol>
     */
    private void resetState() {
        if (icrContext != null) {
            icrContext.clearStrokes();
        }

        currBB   = null;
        currPage = null;

        // HWR

        hwrResult       = null;
        hwrResultBounds = null;

        // Other state

        seenDoubleTap = false;
    }

    // Events

    /**
     * A new stroke was created by the user.
     * <p>
     * This performs the following tasks:</p>
     * <ol>
     * <li>Sets {@link #recordNextStroke} to <code>true</code> and returns
     *     immediately if it was set to <code>false</code>.</li>
     * <li>Notifies the ICR context of the latest pen down time and adds the
     *     stroke.</li>
     * <li>Adds the stroke's bounding box to {@link #currBB}.</li>
     * </ol>
     *
     * @param ev the stroke event
     */
    public boolean onStrokeEvent(StrokeEvent ev) {
        synchronized (lock) {
            if (WAIT_FOR_PEN_DOWN_STATE == state) {
                Log.debug(CLASS_NAME, "onStrokeEvent", "Skipping " + ev);
                return false;
            }

            strokePending = false;

            // Deactivate the plugin if we are marked as done
            // and also if we don't need to record the next stroke

            if (!recordNextStroke) {
                if (FINISHED_STATE == state) {
                    deactivate();
                } else {
                    recordNextStroke = true;
                }
                return true;
            }

            // Process the stroke

            Log.debug(CLASS_NAME, "onStrokeEvent", "Sending stroke to ICRContext");
            icrContext.addStroke(ev.getPageInstance(), ev.getTime());

            // Add the region to the current region

            StrokeStorage strokes = new StrokeStorage(ev.getPageInstance());
            Rectangle bb = strokes.getStrokeBoundingBox(ev.getTime());

            if (currBB == null) {
                currBB = bb;
            } else {
                currBB = Shape.getUnion(currBB, bb);
            }

            return true;
        }
    }

    // Events

    /**
     * Processes a pen tip event.
     * <p>
     * This does not consume double taps if double tap termination is enabled
     * via {@link #setDoubleTapTerminates(boolean)}.  The purpose of passing
     * along double taps is to allow the application to perform some action in
     * case it wishes to alert the user.</p>
     *
     * @param ev the pen tip event
     * @return whether the event was processed.
     */
    public boolean onPenTipEvent(PenTipEvent ev) {
        synchronized (lock) {
            // If we are waiting for pen-down and this is not a pen-down then
            // don't process

            if (WAIT_FOR_PEN_DOWN_STATE == state && PenTipEvent.PEN_TIP_PEN_DOWN != ev.getId()) {
                Log.debug(CLASS_NAME, "onPenTipEvent", "Skipping " + ev);
                return false;
            }

            if (FINISHED_STATE == state) {
                return active;  // There is a race condition between this and
                                // hwrUserPause/hwrResult where calls to this and
                                // the HWR methods occur at almost the same time.
                                // If hwrUserPause/hwrResult is called first, then
                                // there is a chance it will deactivate the plugin
            }

            switch (ev.getId()) {
                case PenTipEvent.PEN_TIP_PEN_DOWN:
                    Log.debug(CLASS_NAME, "onPenTipEvent", "Pen down: region=" + ev.getRegion() + " page=" + ev.getPageInstance());

                    if (WAIT_FOR_PEN_DOWN_STATE == state) {
                        changeState(RUNNING_STATE);
                    }

                    strokePending = true;

                    // Check for a matching page

                    if (currPage != null && currPage.getPageAddress() != ev.getPageInstance().getPageAddress()
                          && !allowPageChange) {
                        //error();
                        resultStatus = RESULT_PAGE_CHANGED;

                        recordNextStroke = false;
                        resetState();

                        // Wait for the stroke to deactivate the plugin

                        changeState(FINISHED_STATE);
                        break;
                    }

                    // Check that we haven't tapped in a known region

                    if (ev.getRegion().getAreaId() != 0 && !allowOverlap) {
                        resultStatus = RESULT_OVERLAPPED;

                        recordNextStroke = false;
                        resetState();

                        // Wait for the stroke to deactivate the plugin

                        changeState(FINISHED_STATE);
                        break;
                    }

                    recordNextStroke = true;
                    Log.debug(CLASS_NAME, "onPenTipEvent", "Sending pen down to ICRContext");
                    icrContext.notifyPenDown(ev.getTime());
                    currPage = ev.getPageInstance();

                    break;

                /*case PenTipEvent.PEN_TIP_SINGLE_TAP:
                    // If the tap happened in one of our regions then display it

                    if (penDownRegion != null && penDownRegion.getAreaId() == 1) {
                        recordNextStroke = false;

                        PropertyCollection pc = PropertyCollection.getInstance(
                                getPenlet().getContext(),
                                penDownPage.getPageAddress() + ".props");

                        if (pc != null) {
                            String result = pc.getProperty(penDownRegion.getId());
                            if (result != null) {
                                result = Strings.decode(result);
                                logger.debug("Decoded string=" + result);
                                player.play(SELECT_SOUND);
                                showResult(result);
                            }
                        }

                        resetState();
                    }
                    break;*/

                case PenTipEvent.PEN_TIP_DOUBLE_TAP:
                    Log.debug(CLASS_NAME, "onPenTipEvent", "Double tap!");

                    // There must be a current bounding box and page

                    if (currBB != null && currPage != null) {
                        // The double tap is probably on a separate engine than the HWR engine,
                        // so do the processing when we've seen a result
                        // This way, we can have the latest stable result

                        seenDoubleTap = true;

                        if (hwrResult != null && doubleTapTerminates) {
                            // Wait for the stroke to deactivate the plugin
                            // so that the lone stroke event doesn't get
                            // passed along to the app

                            changeState(FINISHED_STATE);
                        }

                        if (doubleTapTerminates) {
                            return false;  // Allow the application to do something with the double-tap, even though the plugin is still active
                        }
                    }
                    break;
            }

            return true;
        }
    }

    // Object Methods

    /**
     * Returns a string representation of this object.
     *
     * @return a string representation of this object.
     */
    public String toString() {
        return "Acquire HW Plugin";
    }
}
