/*
 * Date: Jul 14, 2010
 * Time: 5:16:48 PM
 * (c) 2010 Livescribe, Inc.
 */
package com.livescribe.ext.io;

import java.io.IOException;
import java.io.InputStream;
import java.io.DataInputStream;

/**
 * This adds the ability to buffer input.  An internal buffer is refilled as
 * necessary.  This does not implement the <em>mark</em>/<em>reset</em>
 * functionality.
 * <p>
 * This class is helpful, for example, when using a {@link DataInputStream}
 * since that class may use single-byte reads when constructing larger
 * primitive types.  The underlying stream may be poorly buffered.</p>
 *
 * @since 2.5
 *
 * @author Shawn Silverman
 */
public class BufferedInputStream extends InputStream {
    /** The wrapped input stream. */
    protected volatile InputStream in;

    /** The internal buffer. */
    protected volatile byte[] buf;

    /**
     * The number of valid bytes in the buffer, in the range
     * 0&ndash;<code>buf.length</code>.
     */
    protected int count;

    /**
     * The current buffer position, pointing to the next byte to be read from
     * the buffer.  This will be in the range 0&ndash;{@link #count}.  If it
     * is equal to {@link #count} then more bytes will need to be read from
     * the underlying stream for the next <em>read</em> or <em>skip</em> call.
     */
    protected int pos;

    /**
     * Creates a new buffered input stream having the specified buffer size.
     *
     * @param in the input stream
     * @param bufSize the buffer size
     * @throws IllegalArgumentException if the buffer size is &le; zero.
     */
    public BufferedInputStream(InputStream in, int bufSize) {
        if (bufSize <= 0) {
            throw new IllegalArgumentException("Buffer size must be > 0");
        }

        this.in = in;
        this.buf = new byte[bufSize];
    }

    /**
     * Fills the buffer.
     *
     * @pre pos >= count
     * @pre Called from within a synchronized context
     * @throws IOException if there was an error reading from the stream.
     */
    private void fill() throws IOException {
        byte[] b = getBuf();

        int read = getIn().read(b, 0, b.length);
        count = Math.max(0, read);
        pos = 0;
    }

    /**
     * Gets the buffer bute throws an {@link IOException} if the stream is
     * closed.
     *
     * @return the buffer.
     * @throws IOException if the stream is closed.
     */
    private byte[] getBuf() throws IOException {
        byte[] buffer = this.buf;
        if (buffer == null) throw new IOException("Closed");
        return buffer;
    }

    /**
     * Gets the underlying input stream but throws an {@link IOException} if
     * the stream is closed.
     *
     * @return the underlying input stream.
     * @throws IOException if the stream is closed.
     */
    private InputStream getIn() throws IOException {
        InputStream input = this.in;
        if (input == null) throw new IOException("Closed");
        return input;
    }

    public synchronized int read() throws IOException {
        if (pos >= count) {
            fill();
            if (pos >= count) return -1;
        }

        return getBuf()[pos++] & 0xff;
    }

    /**
     * Reads from the underlying stream or buffer as necessary.
     *
     * @param b read into this buffer
     * @param off into the buffer
     * @param len the number of bytes to read
     * @return the actual number of bytes read.
     * @throws IOException if there was a read error.
     */
    private int readInternal(byte[] b, int off, int len) throws IOException {
        int avail = count - pos;

        if (avail <= 0) {
            // If the length is at least as large as the buffer, just read
            // directly from the underlying stream
            // This allows buffered streams to chain more effectively

            if (len >= getBuf().length) {
                return getIn().read(b, off, len);
            } else {
                fill();
                avail = count - pos;
                if (avail <= 0) return -1;
            }
        }

        // Some bytes should now be available
        // Copy them from the internal buffer into the read buffer

        int count = Math.min(avail, len);
        System.arraycopy(getBuf(), pos, b, off, count);
        pos += count;
        return count;
    }

    public synchronized int read(byte[] b, int off, int len) throws IOException {
        getBuf();  // Check if closed

        // Check the arguments

        if (b == null) {
            throw new NullPointerException();
        } else if (off < 0 || len < 0 || off + len < 0 || b.length - (off + len) < 0) {
            throw new IndexOutOfBoundsException();
        } else if (len == 0) {
            return 0;
        }

        int n = 0;

        while (true) {
            int read = readInternal(b, off + n, len - n);
            if (read <= 0) return (n == 0) ? read : n;

            n += read;

            // Check if we need to stop because we've read enough

            if (n >= len) return n;

            // Return if we aren't closed but no more bytes are available

            InputStream input = this.in;

            if (input != null && input.available() <= 0) {
                return n;
            }
        }
    }

    public synchronized long skip(long n) throws IOException {
        getBuf();  // Check if closed

        if (n <= 0) return 0;

        long avail = (count - pos);

        // Check if we need to skip in the underlying stream

        if (avail <= 0) return getIn().skip(n);

        long skipped = Math.min(avail, n);
        pos += skipped;  // TODO long-type problems
        return skipped;
    }

    public synchronized int available() throws IOException {
        return getIn().available() + Math.max(0, count - pos);
    }

    /**
     * Can asynchronously close this stream.
     *
     * @throws IOException if there was an error while closing the stream.
     */
    public void close() throws IOException {
        InputStream input = this.in;
        if (input != null) {
            this.in = null;
            input.close();
        }
    }
}
