package com.livescribe.samples;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Enumeration;

import com.livescribe.afp.PageInstance;
import com.livescribe.display.Display;
import com.livescribe.display.Image;
import com.livescribe.ext.util.Log;
import com.livescribe.i18n.ImageResource;
import com.livescribe.i18n.ResourceBundle;
import com.livescribe.i18n.SoundResource;
import com.livescribe.i18n.TextResource;
import com.livescribe.penlet.PenletContext;
import com.livescribe.penlet.Region;
import com.livescribe.ui.MediaPlayer;
import com.livescribe.ui.ScrollLabel;

/**
 * Helper for tap and play penlets
 * 
 * processPenEvent() will get the area name for the region tapped and search
 * for matching text, image, and audio resources.  If a resource is found, then
 * the text and/or image is displayed.  If a sound is found, then it is played.
 * 
 * Note: Sound files are only processed if a player exists.  
 * Text, image, and sound processing can be disabled by setting the corresponding flag.
 */
public class TapAndPlayHelper {
    private Display display;
    private ScrollLabel label;
    private MediaPlayer player;
    private ResourceBundle resBundle;
    private boolean processSounds;
    private boolean processText;
    private boolean processImages;

    // Used to mask the penlet instance from the region id
    private final static long INSTANCE_MASK = 0xffff000000ffffffL;
    
    /**
     * Constructor for labels, images, and sounds
     * @param context Pass the penlet context
	 * @param player Pass the media player or null to skip sound processing
     */
    public TapAndPlayHelper(PenletContext context, MediaPlayer player) {
    	this.display = context.getDisplay();
    	this.resBundle = context.getResourceBundle();
    	this.label = new ScrollLabel();
    	this.player = player;
    	this.processSounds = (player != null);
        this.processText = true;
        this.processImages = true;
    }

    /**
     * Handle the pen event by displaying text, displaying an image, and/or 
     * playing a sound that matches the area id name for the region passed.
	 * @param region The region in which the event occurred
	 * @param page The page on which the event occurred
	 * @return true if the event was successfully handled, false otherwise
     */
    public boolean processPenEvent(Region region, PageInstance page) {
		boolean handled = false;
        String areaName = getAreaName(region.getId(), page);
        if (areaName != null) {
        	handled = displayTextAndImage(areaName);
        	handled |= playSound(areaName);
        }
		return handled;
	}
    
    /**
     * Display the text and/or image matching the name
     * @param name The name of the resource
     * @return true if anything was displayed, false otherwise
     */
    private boolean displayTextAndImage(String name) {
    	Image img = null;
    	String text = null;

    	if (processImages) {
    		// Look for the image resource
            ImageResource imageRes = resBundle.getImageResource(name);
            if (imageRes != null)
            	img = imageRes.getImage(); 
    	}

    	if (processText) {
            // Look for the text resource
            TextResource textRes = resBundle.getTextResource(name);
	    	if (textRes != null)
	    		text = textRes.getText();
    	}

    	// Display if either exist
    	boolean show = (img != null || text != null);
    	if (show) {
            this.label.draw(text, img, true);
            this.display.setCurrent(this.label);
    	}

    	return show;
    }

    /**
     * Play a sound file matching the file name
     * @param fileName The name of the resource
     * @return true if a sound was played, false otherwise
     */
    private boolean playSound(String fileName) {
    	boolean handled = false;
    	if (processSounds) {
	        SoundResource res = resBundle.getSoundResource(fileName);
	        if (res != null) {
	        	player.play(res);
	        	handled = true;
	        }
    	}
    	return handled;
    }
	
    /**
     * Get the area name for the region and page
	 * @param regionId The region id in which the event occurred
	 * @param page The page on which the event occurred
     * @return Returns the area name or null if not found
     */
    private String getAreaName(long regionId, PageInstance page) {
    	String name = null;
    	
    	// Area names are stored in the auxiliary properties file
    	try {
    		// Load the info into the XML parser
	    	InputStream is = page.getDocument().getResource("/userdata/auxiliary.properties");
	    	XMLElement xml = new XMLElement();
	    	xml.parseFromReader(new InputStreamReader(is, "UTF-8"));
	    	is.close();
	    	
	    	// Find the page node
	    	XMLElement pageNode = findNode(xml, "pt", "id", page.getPage());
	    	if (pageNode != null) {
	    		// Find the area
		    	XMLElement area = findNode(pageNode, "shape", "regionId", regionId);
	    		if (area != null) {
			    	name = area.getStringAttribute("desc");
	    		}
	    	}
    	}
    	catch (IOException e) {
    		Log.error("Error reading auxiliary.properties.  " + e.getMessage());
    	}

    	return name;
    }

    /**
     * Find the node matching the tag, attribute, and value
     * @param xml The XML element to search
     * @param tag The tag to match
     * @param attrib The attribute to match
     * @param value The value to match
     * @return the matching node or null if not found
     */
    private XMLElement findNode(XMLElement xml, String tag, String attrib, long value) {
    	XMLElement node = null;

    	if (xml != null) {
	    	if (tag.equals(xml.getName()) && xml.getStringAttribute(attrib) != null ) {
	    		try {
		    		long attribValue = Long.parseLong(xml.getStringAttribute(attrib));
		    		if (attribValue == value
		    		 || (attrib.equals("regionId") && getMaskedRegion(attribValue) == getMaskedRegion(value))) {
		    			// The node matches
		    			node = xml;
		    		}
	    		}
	    		catch (NumberFormatException e){
	    			// It's not a match if it's not a number
	    		}
	    	}

	    	if (node == null) {
	    		// Search the children for a match
		    	Enumeration kids = xml.enumerateChildren();
	    		while (node == null && kids.hasMoreElements()) {
		    		node = findNode((XMLElement)kids.nextElement(), tag, attrib, value);
		    	}
	    	}
    	}
    	return node;
    }

    /**
     * Zero out the instance (penlet id) from the region id
     * @param id Pass the region id
     * @return Returns the region with the instance masked out
     */
    private long getMaskedRegion(long id) {
    	return id & INSTANCE_MASK;
    }

	/**
	 * @param process Pass true to process sounds, false otherwise
	 */
	public void setProcessSounds(boolean process) {
		this.processSounds = (process && player != null);
	}

	/**
	 * @param process Pass true to display text, false otherwise
	 */
	public void setProcessText(boolean process) {
		this.processText = process;
	}

	/**
	 * @param process Pass true to display images, false otherwise
	 */
	public void setProcessImages(boolean process) {
		this.processImages = process;
	}

}
